#include <stdlib.h>
#include <stdio.h>

#include "common.h"

/*
** Required includes to get instance creation functions for both processes
*/
#include "pSender_decl.h"
#include "pReceiver_decl.h"

/*
** No RTOS, so we have to manage our own message queues. In the context
** of a scheduler, only one system-wide message queue is needed.
*/
typedef struct _SystemQueue
	{
	MessageIdentifier			message_id;
	int										receiver_index;
	} SystemQueue;
#define SYSTEM_QUEUE_CAPACITY 4
static SystemQueue system_queue[SYSTEM_QUEUE_CAPACITY];
static int system_queue_first = 0;
static int system_queue_next = 0;
/*
** We also need messages represented as RTDS standard RTDS_MessageHeader
** variables.
*/
static RTDS_MessageHeader message_headers[SYSTEM_QUEUE_CAPACITY];


/*
** FUNCTION message_send:
** ----------------------
** Sends a message, i.e stores it in the global system queue
*/
void message_send(MessageIdentifier message_id, int receiver_index)
	{
	system_queue[system_queue_next].message_id = message_id;
	system_queue[system_queue_next].receiver_index = receiver_index;
	
	system_queue_next = (system_queue_next + 1) % SYSTEM_QUEUE_CAPACITY;
	}


/*
** MAIN FUNCTION:
** ==============
** Simplified scheduler, creating both instances, then running transitions one by one.
*/
int main(void)
	{
	RTDS_MSG_DATA_DECL
	
	/* Create instances */
	RTDS_Proc_pSender_createInstance(&(process_descriptors[0]), NULL, &(process_contexts[0]), NULL);
	RTDS_Proc_pReceiver_createInstance(&(process_descriptors[1]), NULL, &(process_contexts[1]), NULL);
	/* Execute initial transitions */
	process_descriptors[0].RTDS_executeTransition(&(process_descriptors[0]), NULL);
	process_descriptors[1].RTDS_executeTransition(&(process_descriptors[1]), NULL);
	/*
	** Loop while system queue is not empty.
	** NB: pSender should have sent a message in its initial transition, so system queue is not empty
	** at the first loop.
	*/
	while ( system_queue_first != system_queue_next )
		{
		/* Build RTDS message header for first message in system queue */
		switch(system_queue[system_queue_first].message_id)
			{
			/*
			** Macros RTDS_<message name>_SET_MESSAGE are provided by RTDS in the generated code.
			** Here, they only take a RTDS_MessageHeader* as parameters; if signals had parameters,
			** the macro would accept them as additional parameters after the RTDS_MessageHeader*.
			*/
			case ping_message:
			  RTDS_sPing_SET_MESSAGE(&(message_headers[system_queue_first]));
				break;
			case ack_message:
			  RTDS_sAck_SET_MESSAGE(&(message_headers[system_queue_first]));
				break;
			default:
				return 1;
			}
		/*
		** Execute transition corresponding to the message above. The message receiver is identified
		** by its index in the arrays process_contexts & process_descriptors, which is figured out
		** when the message is sent.
		*/
		switch(system_queue[system_queue_first].receiver_index)
			{
			/* Instance with index 0 is a pSender */
			case 0:
				process_descriptors[0].RTDS_executeTransition(&(process_descriptors[0]), &(message_headers[system_queue_first]));
				break;
			/* Instance with index 1 is a pReceiver */
			case 1:
				process_descriptors[1].RTDS_executeTransition(&(process_descriptors[1]), &(message_headers[system_queue_first]));
				break;
			default:
				return 2;
			}
		/* Next message in system queue */
		system_queue_first = (system_queue_first + 1) % SYSTEM_QUEUE_CAPACITY;
		}
	return 0;
	}

