/* **************************************************************** *
 * Copyright PragmaDev 2003
 * **************************************************************** *
 * This module contains the functions allowing to communicate with
 * the MSC tracer
 * **************************************************************** */

#if defined(__MINGW32__)
#include <Winsock2.h>
#include <ws2tcpip.h>
#else
#include <sys/socket.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#endif

#include <stdio.h>
#include <string.h>

#include "TracerSocketConnection.h"


static int tracerSocket = -1;


/*
 * FUNCTION startTracing:
 * ----------------------
 * Starts the connection to the MSC tracer
 * Parameters:
 * - tracerIPAddress: IP address for the MSC tracer host as a string
 * - portNumber: port number for the tracer
 * Returns: >= 0 if OK; < 0 if KO
 */

int startTracing(char * tracerIPAddress, unsigned short portNumber)
  {
  int iResult;
  /* Socket address structure for server */
  struct sockaddr_in serverAddress;
  
  #if defined(__MINGW32__)
    //----------------------
    // Initialize Winsock
    WSADATA wsaData;
    iResult = WSAStartup(MAKEWORD(2, 2), &wsaData);
    if (iResult != NO_ERROR) {
      printf("WSAStartup function failed with error: %d\n", iResult);
      return 1;
      }
  #endif
  

  /* Build socket address structure for server */
  serverAddress.sin_family = AF_INET;
  serverAddress.sin_addr.s_addr = inet_addr(tracerIPAddress);
  serverAddress.sin_port = htons(portNumber);

  /* Create socket */
  tracerSocket = socket(AF_INET, SOCK_STREAM,0);

  /* Connect to socket */
  iResult = connect(tracerSocket, (struct sockaddr *)&serverAddress, sizeof(struct sockaddr_in));
  
  #if defined(__MINGW32__)
    if (iResult == SOCKET_ERROR) {
      printf("connect function failed with error: %ld\n", WSAGetLastError());
      iResult = closesocket(tracerSocket);
      if (iResult == SOCKET_ERROR)
          printf("closesocket function failed with error: %ld\n", WSAGetLastError());
      WSACleanup();
      return -1;
      }
  #else
    return iResult;
  #endif
  }


/*
 * FUNCTION sendToTracer:
 * ----------------------
 * Sends a tracing command to the MSC tracer
 * Parameters:
 * - command: the command to send
 * Returns: >= 0 if OK; < 0 if KO
 */

int sendToTracer(char * command)
  {
  int iResult;
  
  if ( tracerSocket == -1 ) return -1;
  printf("*** Sending '%s' to tracer...\n", command);
  iResult = send(tracerSocket, command, strlen(command), 0);
  #if defined(__MINGW32__)
  if ( iResult == SOCKET_ERROR ) {
    printf("send function failed with error: %ld\n", WSAGetLastError());
    return -1;
    }
  #endif
  return iResult;
  }


/*
 * FUNCTION waitForAck:
 * --------------------
 * Sends a waitForAck command to the MSC tracer and waits for the answer
 * Parameters: none
 * Returns: >= 0 if OK; < 0 if KO
 */

int waitForAck()
  {
  char buffer[4];
  int cr, nbytes;

  if ( tracerSocket == -1 ) return -1;
  cr = sendToTracer("waitingForAck|\n");
  if ( cr < 0 ) return cr;
  nbytes = recv(tracerSocket, buffer, 3, 0);
  printf("After recv, nbytes= %d\n", nbytes);
  if (nbytes < 0) return nbytes;
  buffer[nbytes] = '\0';
  printf("waitForAck buffer value: %s", buffer);
  if ( strcmp(buffer, "ack") != 0 ) return -1;
  return nbytes;
  }

 
/*
 * FUNCTION endTracing:
 * --------------------
 * Closes the connection to the MSC tracer
 * Parameters: none
 * Returns: >= 0 if OK; < 0 if KO
 */

int endTracing()
  {
  if ( tracerSocket == -1 ) return -1;
  return close(tracerSocket);
  }
