/* Copyright 2013 Humboldt University of Berlin
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program. If not, see <http://www.gnu.org/licenses/>.
*
* Author: Mihal Brumbulli <mbrumbulli@gmail.com>
*/

#include <cstring>
#include <climits>
#include <cstdlib>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "demoddix.h"


std::vector<Tracer>     Tracer::tracerList;
std::vector<pthread_t>  Tracer::threadList;

pthread_t               Tracer::pollThread;
std::string             Tracer::tracerCommand;

bool                    Tracer::doPoll_         = false;
pthread_mutex_t         Tracer::pollLock_;
unsigned int            Tracer::pollInterval    = 200;

// Format for reading values from trace files
static const char       * taskCreatedF      = " <taskCreated nId=\"n%u\" cId=\"c%u\" time=\"%lld\" creatorId=\"%u\" pName=\"p%u\" creatorName=\"p%u\" pId=\"%u\" />";
static const char       * taskDeletedF      = " <taskDeleted nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" pId=\"%u\" />";
static const char       * messageSentF      = " <messageSent nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" mId=\"%u\" pId=\"%u\" sigNum=\"%u\" msgName=\"m%u\" />";
static const char       * messageReceivedF  = " <messageReceived nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" mId=\"%u\" pId=\"%u\" sigNum=\"%u\" msgName=\"m%u\" />";
static const char       * messageSavedF     = " <messageSaved nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" mId=\"%u\" pId=\"%u\" sigNum=\"%u\" msgName=\"m%u\" />";
static const char       * semaphoreCreatedF = " <semaphoreCreated nId=\"n%u\" cId=\"c%u\" time=\"%lld\" semName=\"x%u\" stillAvailable=\"%d\" pId=\"%u\" />";
static const char       * takeAttemptF      = " <takeAttempt nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" semName=\"x%u\" timeout=\"%d\" pId=\"%u\" semId=\"%u\" />";
static const char       * takeSuccededF     = " <takeSucceeded nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" semName=\"x%u\" stillAvailable=\"%d\" pId=\"%u\" semId=\"%u\" />";
static const char       * takeTimedOutF     = " <takeTimedOut nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" semName=\"x%u\" pId=\"%u\" semId=\"%u\" />";
static const char       * giveSemF          = " <giveSem nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" semName=\"x%u\" pId=\"%u\" semId=\"%u\" />";
static const char       * timerStartedF     = " <timerStarted nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" timerName=\"m%u\" pId=\"%u\" tId=\"%u\" timeLeft=\"%d\" />";
static const char       * timerCancelledF   = " <timerCancelled nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" timerName=\"m%u\" pId=\"%u\" tId=\"%u\" />";
static const char       * timerTimedOutF    = " <timerTimedOut nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" timerName=\"m%u\" pId=\"%u\" tId=\"%u\" />";
static const char       * taskChangedStateF = " <taskChangedState nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" pId=\"%u\" stateName=\"s%u\" />";
static const char       * informationF      = " <information nId=\"n%u\" cId=\"c%u\" time=\"%lld\" pName=\"p%u\" pId=\"%u\" message=\"%[^\"]\" />";

// Constructor
Tracer::Tracer()
  {
  pthread_mutex_init(&lock_, NULL);
  pthread_mutex_lock(&lock_);
  port_ = 0;
  sock_ = -1;
  status_ = IDLE; 
  pthread_mutex_unlock(&lock_);
  }

// Destructor
Tracer::~Tracer()
  {
  pthread_mutex_destroy(&lock_);
  }

// Atomic getters and setters
unsigned short Tracer::port()
  { 
  pthread_mutex_lock(&lock_);
  unsigned short value = port_;
  pthread_mutex_unlock(&lock_);
  return value;
  }

void Tracer::port(unsigned short value)
  { 
  pthread_mutex_lock(&lock_);
  port_ = value;
  pthread_mutex_unlock(&lock_);
  }

int Tracer::sock()
  { 
  pthread_mutex_lock(&lock_);
  int value = sock_;
  pthread_mutex_unlock(&lock_);
  return value;
  }

void Tracer::sock(int value)
  { 
  pthread_mutex_lock(&lock_);
  sock_ = value;
  pthread_mutex_unlock(&lock_);
  }

int Tracer::status()
  { 
  pthread_mutex_lock(&lock_);
  int value = status_;
  pthread_mutex_unlock(&lock_);
  return value;
  }

void Tracer::status(int value)
  { 
  pthread_mutex_lock(&lock_);
  status_ = value;
  pthread_mutex_unlock(&lock_);
  }

// Atomic polling
bool Tracer::doPoll() 
  {
  pthread_mutex_lock(&Tracer::pollLock_);
  bool value = Tracer::doPoll_;
  pthread_mutex_unlock(&Tracer::pollLock_);
  return value;
  }

void Tracer::doPoll(bool value)
  { 
  pthread_mutex_lock(&Tracer::pollLock_);
  Tracer::doPoll_ = value;
  pthread_mutex_unlock(&Tracer::pollLock_);
  }

// Initialize
void Tracer::Open() 
  {
  // Look for tracer using environment variables
  char *tracer = getenv("PRAGMADEV_TRACER_COMMAND");
  if (tracer != NULL)
    {
    Tracer::tracerCommand = std::string(tracer) + " --nmw";
    }
  else
    {
    tracer = getenv("RTDS_HOME");
    if (tracer != NULL)
      {
      Tracer::tracerCommand = "pragmatracer --nmw";
      }
    }
  Tracer::tracerList.resize(Demoddix::nodeList.size());
  Tracer::threadList.resize(Demoddix::nodeList.size());
  Tracer::doPoll(true);
  pthread_create(&Tracer::pollThread, NULL, &Tracer::Poll, NULL);
  }

// Terminate
void Tracer::Close() 
  {
  // If not initialized, done
  if (!Tracer::doPoll())
    {
    pthread_mutex_destroy(&Tracer::pollLock_);
    return;
    }
  // Terminate polling thread
  Tracer::doPoll(false);
  pthread_join(Tracer::pollThread, NULL);
  pthread_mutex_destroy(&Tracer::pollLock_);
  // Terminate tracer threads
  for (unsigned long i = 0; i < Tracer::tracerList.size(); ++i)
    {
    if (Tracer::tracerList[i].status() != Tracer::IDLE)
      {
      pthread_join(Tracer::threadList[i], NULL);
      }
    }
  }

// Launch
void Tracer::Launch(unsigned long id) 
  {
  // No tracer, done
  if (Tracer::tracerCommand.empty())
    {
    return;
    }
  // Make sure trace is not already running
  if (Tracer::tracerList[id].status() != Tracer::IDLE) 
    {
    return;
    }
  // Remember used ports from running tracers
  std::vector<unsigned short> used;
  for (unsigned long i = 0; i < Tracer::tracerList.size(); ++i) 
    {
    Tracer &t = Tracer::tracerList[i];
    if (t.port() > 0) 
      {
      used.push_back(t.port());
      }
    }
  // Find a free port
  unsigned short p = USHRT_MAX;
  for (; p > 0; --p) 
    {
    bool exists = false;
    for (unsigned long i = 0; i < used.size(); ++i)
      {
      if (used[i] == p)
        {
        exists = true;
        break;
        }
      }
    if (exists) 
      {
      continue;
      }
    // Create socket
    int sock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (sock < 0)
      {
      continue;
      }
    // Make non-blocking
    int flags = fcntl(sock, F_GETFL, 0);
    if (flags < 0)
      {
      close(sock);
      continue;
      }
    else if (fcntl(sock, F_SETFL, flags | O_NONBLOCK) < 0)
      {
      close(sock);
      continue;
      }
    // Make socket reusable
    int yes = 1;
    if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (const void*) &yes, sizeof(yes)) < 0)
      {
      close(sock);
      continue;
      }
    // Setup address info
    struct sockaddr_in addr;
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(p);
    // Bind
    if (bind(sock, (struct sockaddr *) &addr, sizeof(addr)) < 0)
      {
      close(sock);
      continue;
      }
    close(sock);
    break;
    }
	// No free port
  if (p == 0) 
    {
    std::cerr << "Warning: no free port for tracer on node " << id << "." << std::endl;
    return;
    }
	// Create thread for tracer
  Tracer::tracerList[id].port(p);
  unsigned int *tid = (unsigned int *) malloc(sizeof(unsigned int));
  *tid = id;
  pthread_create(&Tracer::threadList[id], NULL, &Tracer::Create, (void *) tid);
  }

void *Tracer::Create(void *arg)
  {
  unsigned int *id = (unsigned int *) arg;
  Tracer::tracerList[*id].status(Tracer::OPENED);
  char cmd[256];
  sprintf(cmd, "%s -p %d -n Node[%u]", Tracer::tracerCommand.c_str(), Tracer::tracerList[*id].port(), *id);
  system(cmd);
  Tracer::tracerList[*id].status(Tracer::CLOSED);
  free(id);
  return NULL;
  }

// Continously check tracer status
void *Tracer::Poll(void *arg)
  {
  while (Tracer::doPoll()) 
    {
    for (unsigned long id = 0; id < Tracer::tracerList.size(); ++id) 
      {
      switch (Tracer::tracerList[id].status())
        {
        // If tracer is idle, do nothing
        case Tracer::IDLE:
        continue;
        // If tracer is opened, try to connect
        case Tracer::OPENED:
        // Create socket if not done already
        if (Tracer::tracerList[id].sock() < 0)
          {
          Tracer::tracerList[id].sock(socket(AF_INET, SOCK_STREAM, IPPROTO_TCP));
          if (Tracer::tracerList[id].sock() < 0)
            {
            continue;
            }
          }
        // Setup address info
        struct sockaddr_in addr;
        memset(&addr, 0, sizeof(addr));
        addr.sin_family = AF_INET;
        addr.sin_addr.s_addr = inet_addr("127.0.0.1");
        addr.sin_port = htons(Tracer::tracerList[id].port());
				// Connect
				if (connect(Tracer::tracerList[id].sock(), (struct sockaddr *) &addr, sizeof(addr)) == 0) 
          {
          Tracer::tracerList[id].status(Tracer::CONNECTED);
          }
        break;
        // If tracer is connected, do nothing
        case Tracer::CONNECTED:
        break;
        // If tracer is closed, terminate thread
        case Tracer::CLOSED:
        close(Tracer::tracerList[id].sock());
        Tracer::tracerList[id].port(0);
        Tracer::tracerList[id].sock(-1);
        Tracer::tracerList[id].status(Tracer::IDLE);
        pthread_join(Tracer::threadList[id], NULL);
        break;
        }
      }
    // Sleep for some time
    usleep(Tracer::pollInterval * 1000);
    }
  return NULL;
  }

// Format trace and send it to the tracer
bool Tracer::Send(const char* buffer)
  {
  bool isValid = true;
  char command[Demoddix::bufferSize];
  long long time;
  unsigned int nId, cId, creatorId, pId, semId, tId, mId;
  unsigned int pName, creatorName, sigNum, msgName, semName, timerName, stateName;
  int stillAvailable, timeout, timeLeft;
  char info[256];
  if (sscanf(buffer, taskCreatedF, &nId, &cId, &time, &creatorId, &pName, &creatorName, &pId) == 7) 
    {
		sprintf(command, "taskCreated| -t%lld| -c0x%llX| -n%s| -N%s| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, (((unsigned long long) cId) << 32) + creatorId, Demoddix::processList[pName].name.c_str(), Demoddix::processList[creatorName].name.c_str(), (((unsigned long long) cId) << 32) + pId);
    }
  else if (sscanf(buffer, taskDeletedF, &nId, &cId, &time, &pName, &pId) == 5) 
    {
    sprintf(command, "taskDeleted| -t%lld| -n%s| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), (((unsigned long long) cId) << 32) + pId);
    }
  else if (sscanf(buffer, messageSentF, &nId, &cId, &time, &pName, &mId, &pId, &sigNum, &msgName) == 8) 
    {
    sprintf(command, "messageSent| -t%lld| -n%s| -i%u| 0x%llX| %u| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), mId, (((unsigned long long) cId) << 32) + pId, sigNum, Demoddix::messageList[msgName].name.c_str());
    }
  else if (sscanf(buffer, messageReceivedF, &nId, &cId, &time, &pName, &mId, &pId, &sigNum, &msgName) == 8) 
    {
    sprintf(command, "messageReceived| -t%lld| -n%s| -i%u| 0x%llX| %u| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), mId, (((unsigned long long) cId) << 32) + pId, sigNum, Demoddix::messageList[msgName].name.c_str());
    }
  else if (sscanf(buffer, messageSavedF, &nId, &cId, &time, &pName, &mId, &pId, &sigNum, &msgName) == 8) 
    {
    sprintf(command, "messageSaved| -t%lld| -n%s| -i%u| 0x%llX| %u| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), mId, (((unsigned long long) cId) << 32) + pId, sigNum, Demoddix::messageList[msgName].name.c_str());
    }
  else if (sscanf(buffer, semaphoreCreatedF, &nId, &cId, &time, &semName, &stillAvailable, &pId) == 6) 
    {
    sprintf(command, "semaphoreCreated| -t%lld| -s%s| -a%d| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::semaphoreList[semName].name.c_str(), stillAvailable, (((unsigned long long) cId) << 32) + pId);
    }
  else if (sscanf(buffer, takeAttemptF, &nId, &cId, &time, &pName, &semName, &timeout, &pId, &semId) == 8) 
    {
    sprintf(command, "takeAttempt| -t%lld| -n%s| -s%s| -T%d| 0x%llX| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::semaphoreList[semName].name.c_str(), timeout, (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + semId);
    }
  else if (sscanf(buffer, takeSuccededF, &nId, &cId, &time, &pName, &semName, &stillAvailable, &pId, &semId) == 8) 
    {
    sprintf(command, "takeSucceeded| -t%lld| -n%s| -s%s| -a%d| 0x%llX| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::semaphoreList[semName].name.c_str(), stillAvailable, (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + semId);
    }
  else if (sscanf(buffer, takeTimedOutF, &nId, &cId, &time, &pName, &semName, &pId, &semId) == 7) 
    {
    sprintf(command, "takeTimedOut| -t%lld| -n%s| -s%s| 0x%llX| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::semaphoreList[semName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + semId);
    }
  else if (sscanf(buffer, giveSemF, &nId, &cId, &time, &pName, &semName, &pId, &semId) == 7) 
    {
    sprintf(command, "giveSem| -t%lld| -n%s| -s%s| 0x%llX| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::semaphoreList[semName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + semId);
    }
  else if (sscanf(buffer, timerStartedF, &nId, &cId, &time, &pName, &timerName, &pId, &tId, &timeLeft) == 8) 
    {
    sprintf(command, "timerStarted| -t%lld| -n%s| -T%s| 0x%llX| 0x%llX| %d|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::messageList[timerName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + tId, timeLeft);
    }
  else if (sscanf(buffer, timerCancelledF, &nId, &cId, &time, &pName, &timerName, &pId, &tId) == 7) 
    {
    sprintf(command, "timerCancelled| -t%lld| -n%s| -T%s| 0x%llX| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::messageList[timerName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + tId);
    }
  else if (sscanf(buffer, timerTimedOutF, &nId, &cId, &time, &pName, &timerName, &pId, &tId) == 7) 
    {
    sprintf(command, "timerTimedOut| -t%lld| -n%s| -T%s| 0x%llX| 0x%llX|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), Demoddix::messageList[timerName].name.c_str(), (((unsigned long long) cId) << 32) + pId, (((unsigned long long) cId) << 32) + tId);
    }
  else if (sscanf(buffer, taskChangedStateF, &nId, &cId, &time, &pName, &pId, &stateName) == 6) 
    {
    sprintf(command, "taskChangedState| -t%lld| -n%s| 0x%llX| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), (((unsigned long long) cId) << 32) + pId, Demoddix::stateList[stateName].name.c_str());
    }
  else if (sscanf(buffer, informationF, &nId, &cId, &time, &pName, &pId, info) == 6) 
    {
    sprintf(command, "information| -t%lld| -n%s| 0x%llX| %s|\n", 
      (time - Demoddix::beginTime) / 1000000, Demoddix::processList[pName].name.c_str(), (((unsigned long long) cId) << 32) + pId, info);
    }
  else 
    {
    isValid = false;
    }
  // If trace is valid and tracer is connected, try to send via socket
  if (isValid && Tracer::tracerList[nId].status() == Tracer::CONNECTED)
    {
    // If send fails, close socket 
    if (send(Tracer::tracerList[nId].sock(), command, strlen(command), 0) < 0) 
      {
      Tracer::tracerList[nId].status(Tracer::IDLE);
      }
    }
  return isValid;
  }
