#include <stdlib.h>
#include <string.h>

#include "RTDS_Proc.h"

#include "RTDS_gen.h"
#include "RTDS_MACRO.h"


/*
** CONSTRUCTOR:
** ------------
** A process is always created by a scheduler. A procedure is created by its caller.
** - parentScheduler: Parent scheduler for process or procedure.
*/

RTDS_Proc::RTDS_Proc(RTDS_Scheduler * parentScheduler)
  {
  this->RTDS_parentScheduler = parentScheduler;
  this->RTDS_isProcedure = 0;
  this->RTDS_initialMessage = NULL;
  this->RTDS_senderId = NULL;
  this->RTDS_sdlStatePrev = 0;
  this->RTDS_calledProcedure = NULL;
  this->RTDS_nextLabelId = 0;
  this->RTDS_currentContext = NULL;
  }
  
  
/*
** DESTRUCTOR:
** -----------
** A process is always deleted by its scheduler. A procedure is deleted by its caller.
*/
RTDS_Proc::~RTDS_Proc(void)
  {
  /* Delete called procedure if any */
  if ( this->RTDS_calledProcedure != NULL )
    delete this->RTDS_calledProcedure;
  /* Delete message that triggered last executed transition if needed */
  if ( ( this->RTDS_initialMessage != NULL ) && ( this->RTDS_initialMessage != this->RTDS_currentContext->currentMessage ) )
    {
    RTDS_FREE(this->RTDS_initialMessage);
    }
  /* If instance is a process, delete its information in global process information chained list */
  if ( ! this->RTDS_isProcedure )
    {
    RTDS_FORGET_INSTANCE_INFO;
    }
  }


/*
** OPERATION RTDS_msgQueueSendToId:
** --------------------------------
** Sends a message to a process identified by its pid
** - messageNumber: Numerical identifier for message to send.
** - dataLength: Length of data associated to message.
** - pData: Pointer on data associated to message.
** - receiver: Identifier for the receiver process.
*/

void RTDS_Proc::RTDS_msgQueueSendToId(long messageNumber, long dataLength, unsigned char * pData, RTDS_SdlInstanceId * receiver)
  {
  // Forwarded to scheduler
  this->RTDS_parentScheduler->sendMessage(this->RTDS_currentContext, receiver, messageNumber, dataLength, pData);
  }


/*
** OPERATION RTDS_msgQueueSendToEnv:
** ---------------------------------
** Sends a message to the environment.
** - messageNumber: Numerical identifier for message to send.
** - dataLength: Length of data associated to message.
** - pData: Pointer on data associated to message.
*/

void RTDS_Proc::RTDS_msgQueueSendToEnv(long messageNumber, long dataLength, unsigned char * pData)
  {
  // Send message to environment via scheduler
#ifdef RTDS_process_RTDS_Env
  this->RTDS_parentScheduler->sendMessageToName(this->RTDS_currentContext, RTDS_process_RTDS_Env, messageNumber, dataLength, pData);
#endif
  }


/*
** OPERATION RTDS_msgQueueSendToName:
** ----------------------------------
** Sends a message to a process identified by its name
** - messageNumber: Numerical identifier for message to send.
** - dataLength: Length of data associated to message.
** - pData: Pointer on data associated to message.
** - receiverName: Name for the receiver process as a string (ignored)
** - receiverNumber: Identifier for the name of the receiver process.
*/

void RTDS_Proc::RTDS_msgQueueSendToName(long messageNumber, long dataLength, unsigned char * pData, char * receiverName, int receiverNumber)
  {
  // Forwarded to scheduler
  this->RTDS_parentScheduler->sendMessageToName(this->RTDS_currentContext, receiverNumber, messageNumber, dataLength, pData);
  }


/*
** OPERATION RTDS_msgSave:
** -----------------------
** Saves the current message.
** - message: Message to save.
*/

void RTDS_Proc::RTDS_msgSave(RTDS_MessageHeader * message)
  {
  RTDS_MSG_SAVE(message);
  }


/*
** OPERATION RTDS_resetTimer:
** --------------------------
** Resets a running timer.
** - timerNumber: Identifier for the timer to reset.
*/

void RTDS_Proc::RTDS_resetTimer(long timerNumber)
  {
  RTDS_RESET_TIMER(timerNumber);
  }


/*
** OPERATION RTDS_setTimer:
** ------------------------
** Starts a timer.
** - timerNumber: Identifier for the timer to start.
** - delay: Timer after which the timer should fire.
*/

void RTDS_Proc::RTDS_setTimer(long timerNumber, int delay)
  {
  RTDS_SET_TIMER(timerNumber, delay);
  }


/*
** OPERATION RTDS_semaphoreIdTake:
** -------------------------------
** Attempts to take a semaphore with an optional time-out.
** - semaphoreId: Identifier for the semaphore to take.
** - takeTimeOut: Time-out for take, or predefined constant RTDS_SEMAPHORE_TIME_OUT_FOREVER to wait forever
** Returns: 1 on success; 0 on failure
*/

RTDS_SemaphoreStatus RTDS_Proc::RTDS_semaphoreIdTake(RTDS_SemaphoreId semaphoreId, RTDS_SemaphoreTimeout takeTimeOut)
  {
  return RTDS_SEMAPHORE_ID_TAKE(semaphoreId, takeTimeOut);
  }


/*
** OPERATION RTDS_semaphoreNameTake:
** ---------------------------------
** Attempts to take a semaphore identified by its name, with an optional time-out.
** - semaphoreNumber: Numerical identifier for the name of the semaphore to take.
** - takeTimeOut: Time-out for take, or predefined constant RTDS_SEMAPHORE_TIME_OUT_FOREVER to wait forever
** Returns: 1 on success; 0 on failure
*/

RTDS_SemaphoreStatus RTDS_Proc::RTDS_semaphoreNameTake(int semaphoreNumber, RTDS_SemaphoreTimeout takeTimeOut)
  {
  return RTDS_SEMAPHORE_NAME_TAKE("", semaphoreNumber, takeTimeOut);
  }


/*
** OPERATION RTDS_semaphoreIdGive:
** -------------------------------
** Releases a semaphore.
** - semaphoreId: Identifier for the semaphore to give.
*/

void RTDS_Proc::RTDS_semaphoreIdGive(RTDS_SemaphoreId semaphoreId)
  {
  RTDS_SEMAPHORE_ID_GIVE(semaphoreId);
  }


/*
** OPERATION RTDS_semaphoreNameGive:
** ---------------------------------
** Releases a semaphore identified by its name.
** - semaphoreNumber: Numerical identifier for the name of the semaphore to give.
*/

void RTDS_Proc::RTDS_semaphoreNameGive(int semaphoreNumber)
  {
  RTDS_SEMAPHORE_NAME_GIVE("", semaphoreNumber);
  }


/*
** OPERATION RTDS_processCreate:
** -----------------------------
** Asks for the creation of a new process instance. Not used for procedures, that are
** instanciated directly.
** NB:  Restrictions in standard SDL have been enforced with C++ code generation. Hence,
**      a process can only instantiate a process in the same block. So the instantiated
**      one is always managed by the same scheduler as the creating one.
** - processNumber: Numerical identifier for the process.
*/

void RTDS_Proc::RTDS_processCreate(int processNumber)
  {
  // Done via parent scheduler
  this->RTDS_currentContext->offspringSdlInstanceId = this->RTDS_parentScheduler->createInstance(processNumber, this->RTDS_currentContext);
  }


/*
** OPERATION RTDS_setSdlState:
** ---------------------------
** Changes the SDL state for the current process instance or procedure.
** - newState: Numerical identifier for the new state.
*/

void RTDS_Proc::RTDS_setSdlState(int newState)
  {
  this->RTDS_currentContext->sdlState = newState;
  RTDS_SIMULATOR_TRACE(RTDS_sdlStateSet, this->RTDS_currentContext->mySdlInstanceId, newState, this->RTDS_currentContext);
  }
