#ifndef _RTDS_PROC_H_
#define _RTDS_PROC_H_

/* Include for all common types */
#include "RTDS_Common.h"

/* Forward declaration for scheduler, resolved at the end of the file */
class RTDS_Scheduler;

/*
** CLASS RTDS_Proc:
** ----------------
** An instance of this class represents a process or a procedurein the running system
*/

class RTDS_Proc
{
  
  /*
  ** ATTRIBUTES:
  */

  protected:
  
    RTDS_Scheduler          * RTDS_parentScheduler;     // Scheduler for the current process or procedure
    short                     RTDS_isProcedure;         // If true, instance is a procedure
    RTDS_MessageHeader      * RTDS_initialMessage;      // Message that initiated the current transition
    RTDS_SdlInstanceId      * RTDS_senderId;            // Identifier for the instance that sent the last received message
    RTDS_Proc               * RTDS_calledProcedure;     // Procedure called by current one if any
    unsigned int              RTDS_nextLabelId;         // Identifier for the label to branch to on next transition
  
  public:
    RTDS_GlobalProcessInfo  * RTDS_currentContext;      // Context for the process or caller context for a procedure
    int                       RTDS_sdlStatePrev;        // Previous state when executing a transition
    
  
  /*
  ** OPERATIONS:
  */

  public:
  
    /*
    ** CONSTRUCTOR:
    ** ------------
    ** A process is always created by a scheduler. A procedure is created by its caller.
    **  - parentScheduler: Parent scheduler for process or procedure.
    */
    RTDS_Proc(RTDS_Scheduler * parentScheduler);
  
    /*
    ** DESTRUCTOR:
    ** -----------
    ** A process is always deleted by its scheduler. A procedure is deleted by its caller.
    */
    virtual ~RTDS_Proc(void);
    
    /*
    ** OPERATION RTDS_executeTransition:
    ** ---------------------------------
    ** Calls a transition in the current process. The implementation for this method is provided in
    ** each sub-class by the code generation.
    ** - currentMessage: Message sent to the process or procedure.
    ** Returns: 1 if the process has been killed or the procedure has returned; 0 otherwise.
    */
    virtual short RTDS_executeTransition(RTDS_MessageHeader * currentMessage) = 0;
    
    /*
    ** OPERATION RTDS_continuousSignals:
    ** ---------------------------------
    ** Evaluates the conditions for continuous signals in the current state for the process, then
    ** executes all transitions that apply. The implementation for this method is provided in each
    ** sub-class by the code generation.
    ** Parameter:
    **  - lowestPriority: (in/out) the lowest priority for the continuous signal to consider. If
    **                    anything is executed, must be set to the actual priority of the considered
    **                    continuous signal.
    ** Returns: 1 if the process has been killed or the procedure has returned; 0 otherwise.
    */
    virtual short RTDS_continuousSignals(int * lowestPriority) = 0;
    
    /*
    ** OPERATION RTDS_msgQueueSendToId:
    ** --------------------------------
    ** Sends a message to a process identified by its pid
    ** - messageNumber: Numerical identifier for message to send.
    ** - dataLength: Length of data associated to message.
    ** - pData: Pointer on data associated to message.
    ** - receiver: Identifier for the receiver process.
    */
    void RTDS_msgQueueSendToId(long messageNumber, long dataLength, unsigned char * pData, RTDS_SdlInstanceId * receiver);
    
    /*
    ** OPERATION RTDS_msgQueueSendToEnv:
    ** ---------------------------------
    ** Sends a message to the environment.
    ** - messageNumber: Numerical identifier for message to send.
    ** - dataLength: Length of data associated to message.
    ** - pData: Pointer on data associated to message.
    */
    void RTDS_msgQueueSendToEnv(long messageNumber, long dataLength, unsigned char * pData);
    
    /*
    ** OPERATION RTDS_msgQueueSendToName:
    ** ----------------------------------
    ** Sends a message to a process identified by its name
    ** - messageNumber: Numerical identifier for message to send.
    ** - dataLength: Length of data associated to message.
    ** - pData: Pointer on data associated to message.
    ** - receiverName: Name for the receiver process as a string (ignored)
    ** - receiverNumber: Identifier for the name of the receiver process.
    */
    void RTDS_msgQueueSendToName(long messageNumber, long dataLength, unsigned char * pData, char * receiverName, int receiverNumber);
    
    /*
    ** OPERATION RTDS_msgSave:
    ** -----------------------
    ** Saves the current message.
    ** - message: Message to save.
    */
    void RTDS_msgSave(RTDS_MessageHeader * message);
    
    /*
    ** OPERATION RTDS_resetTimer:
    ** --------------------------
    ** Resets a running timer.
    ** - timerNumber: Identifier for the timer to reset.
    */
    void RTDS_resetTimer(long timerNumber);
    
    /*
    ** OPERATION RTDS_setTimer:
    ** ------------------------
    ** Starts a timer.
    ** - timerNumber: Identifier for the timer to start.
    ** - delay: Timer after which the timer should fire.
    */
    void RTDS_setTimer(long timerNumber, int delay);
    
    /*
    ** OPERATION RTDS_semaphoreIdTake:
    ** -------------------------------
    ** Attempts to take a semaphore with an optional time-out.
    ** - semaphoreId: Identifier for the semaphore to take.
    ** - takeTimeOut: Time-out for take, or predefined constant RTDS_SEMAPHORE_TIME_OUT_FOREVER to wait forever
    ** Returns: 1 on success; 0 on failure
    */
    RTDS_SemaphoreStatus RTDS_semaphoreIdTake(RTDS_SemaphoreId semaphoreId, RTDS_SemaphoreTimeout takeTimeOut);
    
    /*
    ** OPERATION RTDS_semaphoreNameTake:
    ** ---------------------------------
    ** Attempts to take a semaphore identified by its name, with an optional time-out.
    ** - semaphoreNumber: Numerical identifier for the name of the semaphore to take.
    ** - takeTimeOut: Time-out for take, or predefined constant RTDS_SEMAPHORE_TIME_OUT_FOREVER to wait forever
    ** Returns: 1 on success; 0 on failure
    */
    RTDS_SemaphoreStatus RTDS_semaphoreNameTake(int semaphoreNumber, RTDS_SemaphoreTimeout takeTimeOut);
    
    /*
    ** OPERATION RTDS_semaphoreIdGive:
    ** -------------------------------
    ** Releases a semaphore.
    ** - semaphoreId: Identifier for the semaphore to give.
    */
    void RTDS_semaphoreIdGive(RTDS_SemaphoreId semaphoreId);
    
    /*
    ** OPERATION RTDS_semaphoreNameGive:
    ** ---------------------------------
    ** Releases a semaphore identified by its name.
    ** - semaphoreNumber: Numerical identifier for the name of the semaphore to give.
    */
    void RTDS_semaphoreNameGive(int semaphoreNumber);
    
    /*
    ** OPERATION RTDS_processCreate:
    ** -----------------------------
    ** Asks for the creation of a new process instance. Not used for procedures, that are
    ** instanciated directly.
    ** - processNumber: Numerical identifier for the process.
    */
    void RTDS_processCreate(int processNumber);
    
    /*
    ** OPERATION RTDS_setSdlState:
    ** ---------------------------
    ** Changes the SDL state for the current process instance or procedure.
    ** - newState: Numerical identifier for the new state.
    */
    void RTDS_setSdlState(int newState);
};


/* Include for scheduler, "forward-declared" above */
#include "RTDS_Scheduler.h"


#else
class RTDS_Proc;
#endif
