#ifdef RTDS_SCHEDULER_DEBUG
#include <stdio.h>
#endif

#include <stdlib.h>

#include "RTDS_MACRO.h"

#include "RTDS_Scheduler.h"
#include "RTDS_InstanceManager.h"


/*
** CONSTRUCTORS:
** ------------
*/

RTDS_Scheduler::RTDS_Scheduler(RTDS_RtosQueueId * schedulerQueue)
  {
  /* Initialize all attributes */
  this->nextInstanceNumber = 0;
  this->scheduledInstances = NULL;
  if ( schedulerQueue == NULL )
    this->externalQueue = RTDS_NEW_MESSAGE_QUEUE;
  else
    this->externalQueue = *schedulerQueue;
  this->internalQueue = NULL;
  this->currentContext = NULL;
  }


/*
** DESTRUCTOR:
** -----------
*/
RTDS_Scheduler::~RTDS_Scheduler(void)
  {
  RTDS_ScheduledInstance  * scheduledInstance2Delete = NULL;
  RTDS_MessageHeader      * message2Delete = NULL;
  
  /* Delete remaining instances */
  while ( this->scheduledInstances != NULL )
    {
    scheduledInstance2Delete = this->scheduledInstances;
    this->scheduledInstances = this->scheduledInstances->next;
    delete scheduledInstance2Delete->instanceObject;
    delete scheduledInstance2Delete;
    }
  /* Delete remaining messages in internal queue */
  while ( this->internalQueue != NULL )
    {
    message2Delete = this->internalQueue;
    this->internalQueue = this->internalQueue->next;
    RTDS_FREE(message2Delete);
    }
  }



/*
** OPERATION run:
** --------------
** Starts the scheduler, i.e actually scheduled the running instances
*/

void RTDS_Scheduler::run(void)
  {
  RTDS_MessageHeader      * currentMessage = NULL;
  RTDS_ScheduledInstance  * prevInstanceListElement = NULL;
  RTDS_ScheduledInstance  * instanceListElement = NULL;
  RTDS_Proc               * receiverInstance = NULL;
  short                     instanceKilled;
  int                       initialState;
  
  /* Message loop */
  for ( ; ; )
    {
    /* If a message is present in the internal message queue, it has priority */
    if ( this->internalQueue != NULL )
      {

#ifdef RTDS_SCHEDULER_DEBUG
      RTDS_MessageHeader * m = this->internalQueue;
      printf("Q: ");
      while ( m != NULL ) { printf("m[%d] ", m->messageNumber); m = m->next; }
      printf("\n");
#endif

      currentMessage = this->internalQueue;
      this->internalQueue = this->internalQueue->next;
      }
    /* If no message in internal message queue, read the next one from the external message queue */
    else
      {
      currentMessage = (RTDS_MessageHeader *)RTDS_MALLOC(sizeof(RTDS_MessageHeader));
      int cr = RTDS_READ_MESSAGE_QUEUE(this->externalQueue, currentMessage);
      if ( cr == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_RECEIVE )
        }
      }
    /* Get actual instance for receiver */
    int receiverInstanceNumber = currentMessage->receiver->instanceNumber;
    receiverInstance = NULL;
    for ( instanceListElement = this->scheduledInstances; instanceListElement != NULL; instanceListElement = instanceListElement->next )
      {
      if ( instanceListElement->instanceNumber == receiverInstanceNumber )
        {
        receiverInstance = instanceListElement->instanceObject;
        break;
        }
      }
    /* If message is a timer time-out and a receiver was found */
    if ( (currentMessage->timerUniqueId != 0) && (receiverInstance != NULL) )
      {
      /* Figure out if the timer was cancelled */
      RTDS_TimerState * current_timer = NULL;
      RTDS_TimerState * prev_timer = NULL;
      for ( current_timer = receiverInstance->RTDS_currentContext->timerList; current_timer != NULL; current_timer = current_timer->next )
        {
        /* If timer found and cancelled */
        if ( current_timer->timerUniqueId == currentMessage->timerUniqueId )
            {
            if ( current_timer->state == RTDS_TIMER_CANCELLED )
                {
                /* Discard current message */
                RTDS_FREE(currentMessage);
                /* Remove it from list of timers */
                if ( prev_timer == NULL )
                    receiverInstance->RTDS_currentContext->timerList = receiverInstance->RTDS_currentContext->timerList->next;
                else
                    prev_timer->next = current_timer->next;
                RTDS_FREE(current_timer);
                /* Forget receiver instance to prevent transition execution */
                receiverInstance = NULL;
                }
            break;
            }
        prev_timer = current_timer;
        }
      }
    /* If a receiver was found */
    if ( receiverInstance != NULL )
      {
      /*
      ** Call the appropriate transition for this message in its receiver instance, considering
      ** the case of a start pseudo-message
      */
      
#ifdef RTDS_SCHEDULER_DEBUG
      printf("i[%d]: s[%d] ->\n", instanceListElement->processNumber, receiverInstance->RTDS_currentContext->sdlState);
#endif

      this->currentContext = receiverInstance->RTDS_currentContext;
      initialState = receiverInstance->RTDS_currentContext->sdlState;
      if ( currentMessage->messageNumber == -1 )
        instanceKilled = receiverInstance->RTDS_executeTransition(NULL);
      else
        {
        RTDS_SIMULATOR_TRACE(
          RTDS_messageReceived,
          currentMessage,
          receiverInstance->RTDS_currentContext->mySdlInstanceId,
          receiverInstance->RTDS_currentContext
        );
        RTDS_RELEASE_MESSAGE_UNIQUE_ID(currentMessage->messageUniqueId);
        instanceKilled = receiverInstance->RTDS_executeTransition(currentMessage);
        }
      /* If instance was not killed, execute transitions for continuous signals in current state */
      if ( ! instanceKilled )
        {
      
#ifdef RTDS_SCHEDULER_DEBUG
        printf("-> s[%d]\n", receiverInstance->RTDS_currentContext->sdlState);
#endif

        /* If a transition changes the process state, continue execution with continuous signals for new state */
        int lowestPriority;
        int previousState = -1;
        while ( ( ! instanceKilled ) && ( receiverInstance->RTDS_currentContext->sdlState != previousState ) )
          {
          previousState = receiverInstance->RTDS_currentContext->sdlState;
          /*
          ** Execute all transition for continuous signals, starting with the one with the highest priority
          ** and until there are no more transitions to execute or instance is killed
          */
          lowestPriority = 0;
          for ( ; ; )
            {
            int previousLowestPriority = lowestPriority;
            /*
            ** If we're in SDL semantics, if there is any message for current instance in the scheduler's queue,
            ** stop considering continuous signals
            */
#ifdef RTDS_SDL_SEMANTICS
            for ( currentMessage = this->internalQueue; currentMessage != NULL; currentMessage = currentMessage->next )
              {
              if ( currentMessage->receiver->instanceNumber == receiverInstanceNumber )
                break;
              }
            if ( currentMessage != NULL )
              break;
#endif
            instanceKilled = receiverInstance->RTDS_continuousSignals(&lowestPriority);
            /* If instance changed its state, was killed, or if no more continuous signal can be executed, over */
            if ( ( receiverInstance->RTDS_currentContext->sdlState != previousState ) || instanceKilled ||
                 ( lowestPriority == previousLowestPriority ) )
              break;
            }
          }
        }
      /* If any executed transition killed the instance */
      if ( instanceKilled )
        {
      
#ifdef RTDS_SCHEDULER_DEBUG
        printf("-> X\n");
#endif

        /* Actually delete the instance */
        delete receiverInstance;
        /* Remove entry from scheduled instances */
        /* NB: browse list again! processes may have been created in the transition so the list may have changed... */
        prevInstanceListElement = NULL;
        for ( instanceListElement = this->scheduledInstances; instanceListElement != NULL; instanceListElement = instanceListElement->next )
          {
          if ( instanceListElement->instanceNumber == receiverInstanceNumber )
            {
            if ( prevInstanceListElement == NULL )
              this->scheduledInstances = this->scheduledInstances->next;
            else
              prevInstanceListElement->next = instanceListElement->next;
            delete instanceListElement;
            break;
            }
          prevInstanceListElement = instanceListElement;
          }
        }
      /* If instance wasn't killed */
      else
        {
        /* If instance state has changed, transfer its write save queue to its read save queue */
        if ( ( receiverInstance->RTDS_currentContext->sdlState != initialState ) &&
             ( receiverInstance->RTDS_currentContext->writeSaveQueue != NULL ) )
          {
          /* Let's get to the end of the save queue */
          currentMessage = receiverInstance->RTDS_currentContext->writeSaveQueue;
          while( currentMessage->next != NULL ) currentMessage = currentMessage->next;
          currentMessage->next = receiverInstance->RTDS_currentContext->readSaveQueue;
          receiverInstance->RTDS_currentContext->readSaveQueue = receiverInstance->RTDS_currentContext->writeSaveQueue;
          receiverInstance->RTDS_currentContext->writeSaveQueue = NULL;
          }
        /* If instance has saved messages */
        if ( receiverInstance->RTDS_currentContext->readSaveQueue != NULL )
          {
          /* Put them at the beginning of the scheduler's internal queue */
          currentMessage = receiverInstance->RTDS_currentContext->readSaveQueue;
          while ( currentMessage->next != NULL ) currentMessage = currentMessage->next;
          currentMessage->next = this->internalQueue;
          this->internalQueue = receiverInstance->RTDS_currentContext->readSaveQueue;
          /* Reset instance's saved messages */
          receiverInstance->RTDS_currentContext->readSaveQueue = NULL;
          }
        }
      }
    }
  }


/*
** OPERATION createInstance:
** -------------------------
** Creates a new instance of a process identified by its number
** Returns: The created process
*/

RTDS_SdlInstanceId * RTDS_Scheduler::createInstance(int processNumber, RTDS_GlobalProcessInfo * creatorContext, RTDS_GlobalProcessInfo * instanceContext)
  {
  RTDS_Proc               * newInstance = NULL;
  RTDS_SdlInstanceId      * newInstanceId = NULL;
  
  /* Create an instance of the class describing the process via RTDS_InstanceManager */
  newInstance = RTDS_InstanceManager::createInstance(this, processNumber);
  /* Create context for instance if needed and register it in scheduler's instances */
  newInstanceId = this->registerInstance(newInstance, processNumber, creatorContext, instanceContext);
  /* If instance context has been created */
  if ( instanceContext == NULL )
    {
    /* Add the process information to the end of the RTDS_globalProcessInfo global list */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalProcessInfo == NULL )
      {
      RTDS_globalProcessInfo = newInstance->RTDS_currentContext;
      }
    else
      {
      /* Let's get to the end of the list */
      RTDS_GlobalProcessInfo * processInfo = RTDS_globalProcessInfo;
      while( processInfo->next != NULL ) processInfo = processInfo->next;
      processInfo->next = newInstance->RTDS_currentContext;
      }
    RTDS_CRITICAL_SECTION_STOP;
    }
  /* Trace for debugger */
  RTDS_SIMULATOR_TRACE(RTDS_processCreated, newInstance->RTDS_currentContext, NULL, creatorContext);
  
  /* Send start pseudo-message to newly created instance */
  this->sendMessage(NULL, newInstanceId, -1, 0, NULL);
  
  return newInstanceId;
  }

RTDS_SdlInstanceId * RTDS_Scheduler::createInstance(int processNumber)
  {
  return this->createInstance(processNumber, NULL);
  }


/*
** OPERATION registerInstance:
** ---------------------------
** Registers an instance of a process, optionnally creating its context
** Parameters:
**  - newInstanceId: The newly created process instance.
**  - processNumber: The numerical identifier for the process name.
**  - creatorContext: The context for the instance creator.
**  - instanceContext: The context for the instance itself if it already exists.
** Returns the SDL instance identifier for the instance.
*/

RTDS_SdlInstanceId * RTDS_Scheduler::registerInstance(
  RTDS_Proc               * newInstance,
  int                       processNumber,
  RTDS_GlobalProcessInfo  * creatorContext,
  RTDS_GlobalProcessInfo  * instanceContext
)
  {
  RTDS_SdlInstanceId      * newInstanceId = NULL;
  RTDS_ScheduledInstance  * scheduledInstance = NULL;
  
  /* Create identification descriptor for new instance */
  newInstanceId = new RTDS_SdlInstanceId;
  /* If no context for instance yet */
  if ( instanceContext == NULL )
    {
    /* Get new instance number in scheduler */
    newInstanceId->queueId = this->externalQueue;
    newInstanceId->instanceNumber = this->nextInstanceNumber;
    this->nextInstanceNumber ++;
    /* Create a new context for instance and initialize it */
    instanceContext = (RTDS_GlobalProcessInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalProcessInfo));
    instanceContext->sdlProcessNumber = processNumber;
    instanceContext->mySdlInstanceId = newInstanceId;
    if ( creatorContext == NULL )
      instanceContext->parentSdlInstanceId = NULL;
    else
      instanceContext->parentSdlInstanceId = creatorContext->mySdlInstanceId;
    instanceContext->offspringSdlInstanceId = NULL;
    instanceContext->sdlState = 0;
    instanceContext->currentMessage = NULL;
    instanceContext->timerList = NULL;
    instanceContext->readSaveQueue = NULL;
    instanceContext->writeSaveQueue = NULL;
    instanceContext->next = NULL;
    }
  /* If there is already a context, get all information from it */
  else
    {
    newInstanceId->queueId = instanceContext->mySdlInstanceId->queueId;
    newInstanceId->instanceNumber = instanceContext->mySdlInstanceId->instanceNumber;
    }
  /* Record context in process instance */
  newInstance->RTDS_currentContext = instanceContext;
  /* Record process descriptor in running instances */
  scheduledInstance = new RTDS_ScheduledInstance;
  scheduledInstance->processNumber = processNumber;
  scheduledInstance->instanceNumber = newInstanceId->instanceNumber;
  scheduledInstance->instanceObject = newInstance;
  scheduledInstance->next = this->scheduledInstances;
  this->scheduledInstances = scheduledInstance;
  
  return newInstanceId;
  }


/*
** OPERATION sendMessage:
** ----------------------
** Sends a message to a given process instance
** - senderContext: The context for the process instance sending the message
** - receiver: The process instance receiving the message
** - messageNumber: Internal number for message type to send
** - dataLength: Length of data associated with the sent message
** - pData: Pointer of the memory zone containing the message data
*/

void RTDS_Scheduler::sendMessage(RTDS_GlobalProcessInfo * senderContext, RTDS_SdlInstanceId * receiver,
                                 int messageNumber, long dataLength, void * pData)
  {
  RTDS_MessageHeader  * newMessage = NULL;
  
  /* If receiver is not managed by this scheduler */
  if ( receiver->queueId != this->externalQueue )
    {
    /* Get context for sender with the correct variable name */
    RTDS_GlobalProcessInfo * RTDS_currentContext = senderContext;
    /* Use standard macro to send the message */
    RTDS_MSG_QUEUE_SEND_TO_ID(messageNumber, dataLength, pData, receiver);
    /* Over */
    return;
    }
  /* If receiver is managed by this scheduler, allocate message descriptor for new message in internal queue */
  if ( this->internalQueue == NULL )
    {
    this->internalQueue = (RTDS_MessageHeader *)RTDS_MALLOC(sizeof(RTDS_MessageHeader));
    newMessage = this->internalQueue;
    }
  else
    {
    newMessage = this->internalQueue;
    while ( newMessage->next != NULL )
      newMessage = newMessage->next;
    newMessage->next = (RTDS_MessageHeader *)RTDS_MALLOC(sizeof(RTDS_MessageHeader));
    newMessage = newMessage->next;
    }
  /* Record all needed information in descriptor */
  newMessage->messageNumber = messageNumber;
  newMessage->timerUniqueId = 0;
  #ifdef RTDS_SIMULATOR
    if (messageNumber != -1)
      newMessage->messageUniqueId = RTDS_GET_MESSAGE_UNIQUE_ID;
    else
      newMessage->messageUniqueId = 0; /* No unique id when start message */
  #endif
  if ( senderContext == NULL )
    newMessage->sender = NULL;
  else
    newMessage->sender = senderContext->mySdlInstanceId;
  newMessage->receiver = receiver;
  newMessage->dataLength = dataLength;
  newMessage->pData = (unsigned char *)pData;
  newMessage->next = NULL;
  /* Trace */
  if (messageNumber != -1)
    {
    RTDS_SIMULATOR_TRACE(RTDS_messageSent, newMessage, receiver, senderContext);
    }
  }


/*
** OPERATION sendMessageToName:
** ----------------------------
** Sends a message to a process instance knowing only the process name
** - senderContext: The context for the process instance sending the message
** - receiverNameNumber: Internal identifier for the receiver process
** - messageNumber: Internal number for message type to send
** - dataLength: Length of data associated with the sent message
** - pData: Pointer of the memory zone containing the message data
*/

void RTDS_Scheduler::sendMessageToName(RTDS_GlobalProcessInfo * senderContext, int receiverNameNumber,
                                       int messageNumber, long dataLength, void * pData)
  {
  RTDS_ScheduledInstance  * instanceInfo = NULL;
  
  /* Look for a process instance having the given name */
  for ( instanceInfo = this->scheduledInstances; instanceInfo != NULL; instanceInfo = instanceInfo->next )
    {
    if ( instanceInfo->processNumber == receiverNameNumber )
      {
      /* If found, send the message to it */
      this->sendMessage(senderContext, instanceInfo->instanceObject->RTDS_currentContext->mySdlInstanceId, messageNumber, dataLength, pData);
      return;
      }
    }
  /* If not found, process is not managed by this scheduler => use standard macro */
  RTDS_GlobalProcessInfo * RTDS_currentContext = senderContext;
  RTDS_MSG_QUEUE_SEND_TO_NAME(messageNumber, dataLength, pData, "", receiverNameNumber);
  }
