#ifndef _RTDS_SCHEDULER_H_
#define _RTDS_SCHEDULER_H_

#include <stdlib.h>

#include "RTDS_ADDL_MACRO.h"

#include "RTDS_OS.h"

#include "RTDS_Proc.h"


/*
** TYPE RTDS_ScheduledInstance:
** ----------------------------
** Records a process instance internally in a scheduler
*/

typedef struct _RTDS_ScheduledInstance
  {
  int                               processNumber;  // Numerical identifier for instantiated process
  unsigned long                     instanceNumber; // Numerical identifier for instance
  RTDS_Proc                       * instanceObject; // Object for scheduled instance
  struct _RTDS_ScheduledInstance  * next;           // Next item if used in chained list
  } RTDS_ScheduledInstance;


/*
** CLASS RTDS_Scheduler:
** =====================
** An instance of this class is a scheduler for several RTDS processes, usually
** all the instances of a given process or all processes in a given block.
*/

class RTDS_Scheduler
{
  /*
  ** ATTRIBUTES:
  */
  
  protected:
  
    unsigned int              nextInstanceNumber; // Numerical identifier for next created instance
    RTDS_ScheduledInstance  * scheduledInstances; // Chained list of contexts for all process instances running in scheduler
    RTDS_RtosQueueId          externalQueue;      // Queue for messages coming from the outside of the scheduler
    RTDS_MessageHeader      * internalQueue;      // Chained list for all messages in scheduler's internal queue
  
  public:
  
    RTDS_GlobalProcessInfo  * currentContext;     // Context for the currently running process
    
  /*
  ** OPERATIONS:
  */
  
  public:
  
    /*
    ** CONSTRUCTORS:
    ** -------------
    */
    RTDS_Scheduler(RTDS_RtosQueueId * schedulerQueue = NULL);
  
    /*
    ** DESTRUCTOR:
    ** -----------
    */
    ~RTDS_Scheduler(void);

    /*
    ** OPERATION run:
    ** --------------
    ** Starts the scheduler, i.e actually schedule the running instances
    */
    void run(void);
    
    /*
    ** OPERATION sendMessage:
    ** ----------------------
    ** Sends a message to a given process instance
    ** - sender: The process instance sending the message
    ** - receiver: The process instance receiving the message
    ** - messageNumber: Internal number for message type to send
    ** - dataLength: Length of data associated with the sent message
    ** - pData: Pointer of the memory zone containing the message data
    */
    void sendMessage(RTDS_GlobalProcessInfo * senderContext, RTDS_SdlInstanceId * receiver,
                     int messageNumber, long dataLength, void * pData);
                             
    /*
    ** OPERATION sendMessageToName:
    ** ----------------------------
    ** Sends a message to a process instance knowing only the process name
    ** - sender: The process instance sending the message
    ** - receiverNameNumber: Internal identifier for the receiver process
    ** - messageNumber: Internal number for message type to send
    ** - dataLength: Length of data associated with the sent message
    ** - pData: Pointer of the memory zone containing the message data
    */
    void sendMessageToName(RTDS_GlobalProcessInfo * senderContext, int receiverNameNumber,
                           int messageNumber, long dataLength, void * pData);
  
    
    /*
    ** OPERATION createInstance:
    ** -------------------------
    ** Creates a new instance of a process identified by its number
    **  - processNumber: Identifier for the process to instantiate
    **  - creator: Instance identifier for the creator process. If not specified, no creator.
    **  - instanceContext: Current context for the created instance if known before instance creation.
    ** Returns: The created process
    */
    RTDS_SdlInstanceId * createInstance(int processNumber);
    RTDS_SdlInstanceId * createInstance(int processNumber, RTDS_GlobalProcessInfo * creatorContext, RTDS_GlobalProcessInfo * instanceContext = NULL);


    /*
    ** OPERATION registerInstance:
    ** ---------------------------
    ** Registers an instance of a process, optionnally creating its context
    ** Parameters:
    **  - newInstanceId: The newly created process instance.
    **  - processNumber: The numerical identifier for the process name.
    **  - creatorContext: The context for the instance creator.
    **  - instanceContext: The context for the instance itself if it already exists.
    */
    RTDS_SdlInstanceId * registerInstance(RTDS_Proc * newInstance, int processNumber,
                                          RTDS_GlobalProcessInfo * creatorContext, RTDS_GlobalProcessInfo * instanceContext);
};


/*
** MACRO RTDS_SCHEDULER_NEW:
** -------------------------
** Creates a new scheduler; May be redefined for special behaviors, e.g wholly scheduled systems.
*/
#ifndef RTDS_SCHEDULER_NEW
#define RTDS_SCHEDULER_NEW new RTDS_Scheduler()
#endif

#else
class RTDS_Scheduler;
#endif
