#include <stdlib.h>
#include <string.h>

#include "RTDS_MACRO.h"
#include "RTDS_OS.h"

#include "RTDS_InternalConstants.h"

#include "RTDS_BinarySemaphoreProcess_decl.h"
#include "RTDS_CountingSemaphoreProcess_decl.h"
#include "RTDS_MutexSemaphoreProcess_decl.h"
  
  
/*
** GLOBAL VARIABLES:
** =================
*/

/* RTDS_globalTimerListFirst: head of list of all timers active in the system */
static RTDS_TimerState * RTDS_globalTimerListFirst = NULL;

/* RTDS_globalTimerListCurrent: first element in list of all timers that has not timed-out */
static RTDS_TimerState * RTDS_globalTimerListCurrent = NULL;

/* Pool for message unique identifiers (only in debug) */
#ifdef RTDS_SIMULATOR
void * RTDS_globalMessageUniqueIdPool = NULL;
#endif

/* Global variables for trace entry and encoded message parameters when needed */
#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACE ) || defined( RTDS_FORMAT_TRACE )
RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError , NULL , 0 };
char *RTDS_globalPrintableParameters = NULL;
#endif


/*
** Macros and function for semaphore allocation:
** ---------------------------------------------
*/

/* If there's a configured maximum number of semaphores */
#ifdef RTDS_MAX_SEMAPHORES
/* If this number is 0, semaphores are not used at all */
#if RTDS_MAX_SEMAPHORES != 0
#define RTDS_HAS_SEMAPHORES
/* If defined and not 0, semaphore allocation is static: define static array and functions simulating a malloc & a free */
static char RTDS_semaphores_initialized = 0;
static RTDS_GlobalSemaphoreInfo RTDS_semaphore_info_array[RTDS_MAX_SEMAPHORES];
static char RTDS_semaphore_info_used[RTDS_MAX_SEMAPHORES];
static RTDS_Proc RTDS_semaphore_proc_array[RTDS_MAX_SEMAPHORES];
static char RTDS_semaphore_proc_used[RTDS_MAX_SEMAPHORES];

static RTDS_GlobalSemaphoreInfo * RTDS_allocate_semaphore_info(void)
  {
  int i;
  
  if ( ! RTDS_semaphores_initialized )
    {
    for ( i = 0; i < RTDS_MAX_SEMAPHORES; i++ )
      {
      RTDS_semaphore_info_used[i] = 0;
      RTDS_semaphore_proc_used[i] = 0;
      }
    RTDS_semaphores_initialized = 1;
    }
  for ( i = 0; i < RTDS_MAX_SEMAPHORES; i++ )
    {
    if ( ! RTDS_semaphore_info_used[i] )
      {
      RTDS_semaphore_info_used[i] = 1;
      return & RTDS_semaphore_info_array[i];
      }
    }
  return NULL;
  }

static void RTDS_free_semaphore_info(RTDS_GlobalSemaphoreInfo * semaphore_info)
  {
  int i;
  for ( i = 0; i < RTDS_MAX_SEMAPHORES; i++ )
    {
    if ( & RTDS_semaphore_info_array[i] == semaphore_info )
      {
      RTDS_semaphore_info_used[i] = 0;
      return;
      }
    }
  RTDS_SYSTEM_ERROR(4);
  }

static RTDS_Proc * RTDS_allocate_semaphore_proc(void)
  {
  int i;
  
  if ( ! RTDS_semaphores_initialized )
    {
    for ( i = 0; i < RTDS_MAX_SEMAPHORES; i++ )
      {
      RTDS_semaphore_info_used[i] = 0;
      RTDS_semaphore_proc_used[i] = 0;
      }
    RTDS_semaphores_initialized = 1;
    }
  for ( i = 0; i < RTDS_MAX_SEMAPHORES; i++ )
    {
    if ( ! RTDS_semaphore_proc_used[i] )
      {
      RTDS_semaphore_proc_used[i] = 1;
      return & RTDS_semaphore_proc_array[i];
      }
    }
  return NULL;
  }

static void RTDS_free_semaphore_proc(RTDS_Proc * semaphore_proc)
  {
  int i;
  for ( i = 0; i < RTDS_MAX_SEMAPHORES; i++ )
    {
    if ( & RTDS_semaphore_proc_array[i] == semaphore_proc )
      {
      RTDS_semaphore_proc_used[i] = 0;
      return;
      }
    }
  RTDS_SYSTEM_ERROR(4);
  }
  
#define RTDS_NEW_SEMAPHORE_INFO RTDS_allocate_semaphore_info()
#define RTDS_FREE_SEMAPHORE_INFO(SEMAPHORE_INFO) RTDS_free_semaphore_info(SEMAPHORE_INFO)

#define RTDS_NEW_SEMAPHORE_PROC RTDS_allocate_semaphore_proc()
#define RTDS_FREE_SEMAPHORE_PROC(SEMAPHORE_PROC) RTDS_free_semaphore_proc(SEMAPHORE_PROC)

#endif

/* If there is no configured maximum number of semaphores */
#else

/* If dynamic memory allocation is allowed */
#ifdef RTDS_MALLOC
#define RTDS_HAS_SEMAPHORES

/* Creating a semaphore dynamically allocates a descriptor */
#define RTDS_NEW_SEMAPHORE_INFO (RTDS_GlobalSemaphoreInfo *)RTDS_MALLOC(sizeof(RTDS_GlobalSemaphoreInfo))
#define RTDS_FREE_SEMAPHORE_INFO(SEMAPHORE_INFO) RTDS_FREE(SEMAPHORE_INFO)

#define RTDS_NEW_SEMAPHORE_PROC (RTDS_Proc *)RTDS_MALLOC(sizeof(RTDS_Proc))
#define RTDS_FREE_SEMAPHORE_PROC(SEMAPHORE_PROC) RTDS_FREE(SEMAPHORE_PROC)

/* If dynamic memory allocation is not allowed, it's an error */
#else
#error ERROR! Dynamic memory allocation is not available (no RTDS_MALLOC macro), but no maximum number of semaphores defined via the RTDS_MAX_SEMAPHORES macro!
#endif
#endif



/*
** Functions for message header allocation & liberation if needed:
** ---------------------------------------------------------------
*/
#ifdef RTDS_MAX_MESSAGES

static char RTDS_message_headers_initialized = 0;
static RTDS_MessageHeader RTDS_message_header_array[RTDS_MAX_MESSAGES];
static char RTDS_message_header_used[RTDS_MAX_MESSAGES];

RTDS_MessageHeader * RTDS_allocate_message_header(void)
  {
  int i;
  
  if ( ! RTDS_message_headers_initialized )
    {
    for ( i = 0; i < RTDS_MAX_MESSAGES; i++ ) RTDS_message_header_used[i] = 0;
    RTDS_message_headers_initialized = 1;
    }
  for ( i = 0; i < RTDS_MAX_MESSAGES; i++ )
    {
    if ( ! RTDS_message_header_used[i] )
      {
      RTDS_message_header_used[i] = 1;
      return & RTDS_message_header_array[i];
      }
    }
  return NULL;
  }

void RTDS_free_message_header(RTDS_MessageHeader * message_header)
  {
  int i;
  for ( i = 0; i < RTDS_MAX_MESSAGES; i++ )
    {
    if ( & RTDS_message_header_array[i] == message_header )
      {
      RTDS_message_header_used[i] = 0;
      return;
      }
    }
  RTDS_SYSTEM_ERROR(4);
  }
  
#endif



/*
** Functions for process id & descriptor allocation & liberation if needed:
** ------------------------------------------------------------------------
*/
#ifdef RTDS_MAX_INSTANCES

static char RTDS_instances_initialized = 0;

static RTDS_GlobalProcessInfo RTDS_instance_info_array[RTDS_MAX_INSTANCES];
static char RTDS_instance_info_used[RTDS_MAX_INSTANCES];

static RTDS_SdlInstanceId RTDS_instance_id_array[RTDS_MAX_INSTANCES];
static char RTDS_instance_id_used[RTDS_MAX_INSTANCES];

RTDS_GlobalProcessInfo * RTDS_allocate_instance_info(void)
  {
  int i;
  
  if ( ! RTDS_instances_initialized )
    {
    for ( i = 0; i < RTDS_MAX_INSTANCES; i++ )
      {
      RTDS_instance_info_used[i] = 0;
      RTDS_instance_id_used[i] = 0;
      }
    RTDS_instances_initialized = 1;
    }
  for ( i = 0; i < RTDS_MAX_INSTANCES; i++ )
    {
    if ( ! RTDS_instance_info_used[i] )
      {
      RTDS_instance_info_used[i] = 1;
      return & RTDS_instance_info_array[i];
      }
    }
  return NULL;
  }

void RTDS_free_instance_info(RTDS_GlobalProcessInfo * instance_info)
  {
  int i;
  for ( i = 0; i < RTDS_MAX_INSTANCES; i++ )
    {
    if ( & RTDS_instance_info_array[i] == instance_info )
      {
      RTDS_instance_info_used[i] = 0;
      return;
      }
    }
  RTDS_SYSTEM_ERROR(4);
  }

RTDS_SdlInstanceId * RTDS_allocate_instance_id(void)
  {
  int i;
  
  if ( ! RTDS_instances_initialized )
    {
    for ( i = 0; i < RTDS_MAX_INSTANCES; i++ )
      {
      RTDS_instance_info_used[i] = 0;
      RTDS_instance_id_used[i] = 0;
      }
    RTDS_instances_initialized = 1;
    }
  for ( i = 0; i < RTDS_MAX_INSTANCES; i++ )
    {
    if ( ! RTDS_instance_id_used[i] )
      {
      RTDS_instance_id_used[i] = 1;
      return & RTDS_instance_id_array[i];
      }
    }
  return NULL;
  }

void RTDS_free_instance_id(RTDS_SdlInstanceId * instance_id)
  {
  int i;
  for ( i = 0; i < RTDS_MAX_INSTANCES; i++ )
    {
    if ( & RTDS_instance_id_array[i] == instance_id )
      {
      RTDS_instance_id_used[i] = 0;
      return;
      }
    }
  RTDS_SYSTEM_ERROR(4);
  }

#endif


/*
** Functions for timer allocation & liberation if needed:
** ------------------------------------------------------
*/
/* If there's a configured maximum number of timers and timers are used */
#if defined(RTDS_MAX_TIMERS) && defined(RTDS_HAS_TIMERS)

/* Timer allocation is static: define static array and functions simulating a malloc & a free */
static char RTDS_timers_initialized = 0;
static RTDS_TimerState RTDS_timer_array[RTDS_MAX_TIMERS];
static char RTDS_timer_used[RTDS_MAX_TIMERS];

RTDS_TimerState * RTDS_allocate_timer_state(void)
  {
  int i;
  
  if ( ! RTDS_timers_initialized )
    {
    for ( i = 0; i < RTDS_MAX_TIMERS; i++ ) RTDS_timer_used[i] = 0;
    RTDS_timers_initialized = 1;
    }
  for ( i = 0; i < RTDS_MAX_TIMERS; i++ )
    {
    if ( ! RTDS_timer_used[i] )
      {
      RTDS_timer_used[i] = 1;
      return & RTDS_timer_array[i];
      }
    }
  return NULL;
  }

void RTDS_free_timer_state(RTDS_TimerState * timer_state)
  {
  int i;
  for ( i = 0; i < RTDS_MAX_TIMERS; i++ )
    {
    if ( & RTDS_timer_array[i] == timer_state )
      {
      RTDS_timer_used[i] = 0;
      return;
      }
    }
  RTDS_SYSTEM_ERROR(4);
  }

#endif


/*
** FUNCTION RTDS_MsgQCreate:
** -------------------------
** Supposed to create and return a new queue for a scheduler process. Since
** no RTOS is present, this doesn't create anything and returns NULL.
*/

RTDS_QCB * RTDS_MsgQCreate( void )
  {
  return (RTDS_QCB*)NULL;
  }


  
/*
** FUNCTION RTDS_SystemTick:
** -------------------------
** Called whenever the time increments. Should be called by an interrupt.
*/

static unsigned long RTDS_globalSystemTime = 0;
void RTDS_SystemTick(void)
  {
  /* Increment current time */
  RTDS_globalSystemTime++;
  /* If there is a current timer */
  if ( RTDS_globalTimerListCurrent != NULL )
    {
    /* Decrement its delay */
    RTDS_globalTimerListCurrent->timeoutDelay --;
    /* Make timer time-out if needed, along with all the ones with the same time-out time */
    while ( RTDS_globalTimerListCurrent->timeoutDelay == 0 )
      {
      RTDS_globalTimerListCurrent = RTDS_globalTimerListCurrent->next;
      if ( RTDS_globalTimerListCurrent == NULL ) break;
      }
    }
  }
  
  
/*
** FUNCTION RTDS_incomingSdlEvent:
** -------------------------------
** Default implementation for function handling external events.
** Parameters:
**  - message: pointer on a preallocated message header receiving the next external message to handle if any.
**  - timeLeft: time before the next timer will time-out, in system ticks. If -1, no timer is running.
** Return value:
**  - 1 to indicate there is a message to handle, and that it has been copied to the message parameter;
**  - 0 to indicate that there is no message to handle;
**  - -1 to indicate there is no message, and that the scheduler should get out of its message treatment loop
**    and return control to its caller.
*/

#ifdef RTDS_HANDLE_EXTERNAL_EVENTS
short RTDS_incomingSdlEvent(RTDS_MessageHeader * message, long timeLeft)
  {
  /*
  ** The default implementation is to increment the system time automatically.
  ** This should *NOT* be done in production systems, where the system time is
  ** supposed to be incremented via an interrupt, and not handled here. This
  ** default implementation is just to ensure that systems running on a host
  ** work correctly, especially timers...
  */
  RTDS_SystemTick();
  return 0;
  }
#endif


/*
** FUNCTION RTDS_Wait4ExternalMessage:
** -----------------------------------
** Waits for an external message; called each time the scheduler has
** exhausted all its internal messages.
** An external message is either a message coming from the environment,
** usually triggered by an interrupt, or a message due to a timer time-out.
** Parameters:
** - message: pointer on a pre-allocated message
** Returns: RTDS_OK if the message has been correctly read, RTDS_ERROR if
** the reading fails for any reason.
*/

int RTDS_Wait4ExternalMessage(RTDS_MessageHeader * message)
  {
  RTDS_TimerState * currentTimer = NULL;
  RTDS_TimerState * previousTimer;
  long              timeLeft = -1;
  short             incomingMessage;
  
  /* Wait for an event until one happens */
  for ( ; ; )
    {
#ifdef RTDS_HAS_TIMERS
    /* We'll access the global timer list: make sure it won't get updated */
    RTDS_DISABLE_INTERRUPTS;
    /* First external messages to consider: timers => discard all cancelled ones */
    previousTimer = NULL;
    currentTimer = RTDS_globalTimerListFirst;
    while ( currentTimer != NULL )
      {
      if ( currentTimer->state == RTDS_TIMER_CANCELLED )
        {
        RTDS_TimerState * nextTimer = currentTimer->next;
        if ( previousTimer == NULL )
          RTDS_globalTimerListFirst = nextTimer;
        else
          previousTimer->next = nextTimer;
        if ( RTDS_globalTimerListCurrent == currentTimer)
          RTDS_globalTimerListCurrent = nextTimer;
        RTDS_FREE_TIMER_STATE(currentTimer);
        currentTimer = nextTimer;
        }
      else
        {
        previousTimer = currentTimer;
        currentTimer = currentTimer->next;
        }
      }
    /* If there is a timed-out timer */
    if ( ( RTDS_globalTimerListFirst != NULL ) && ( RTDS_globalTimerListFirst->timeoutDelay == 0 ) )
      {
      /* Get timed out timer */
      currentTimer = RTDS_globalTimerListFirst;
      RTDS_globalTimerListFirst = RTDS_globalTimerListFirst->next;
      if ( RTDS_globalTimerListCurrent == currentTimer )
        RTDS_globalTimerListCurrent = RTDS_globalTimerListCurrent->next;
      /* Build message for timer */
      message->messageNumber = currentTimer->timerNumber;
      message->timerUniqueId = currentTimer->timerUniqueId;
      message->sender = currentTimer->receiverId;
      message->receiver = currentTimer->receiverId;
      message->dataLength = 0;
      message->pData = NULL;
      message->next = NULL;
      /* Free timer descriptor */
      RTDS_FREE_TIMER_STATE(currentTimer);
      /* Over */
      RTDS_ENABLE_INTERRUPTS;
      return RTDS_OK;
      }
    /* If there is no timed-out timer, call function getting next external SDL message */
    if ( RTDS_globalTimerListCurrent != NULL )
      timeLeft = RTDS_globalTimerListCurrent->timeoutDelay;
    RTDS_ENABLE_INTERRUPTS;
#endif
    /* If function requests a break, do it */
    incomingMessage = RTDS_incomingSdlEvent(message, timeLeft);
    if ( incomingMessage == -1 )
      return RTDS_BREAK;
    /* If function actually gets a message, over */
    if ( incomingMessage )
      return RTDS_OK;
    }
  return RTDS_ERROR;
  }


/*
** FUNCTION RTDS_DummyTraceFunction:
** -----------------------------
** Used for tracing: The simulator sets a breakpoint on this function and reads
** the RTDS_globalTraceEntry global variable to see what happened.
*/

#ifdef RTDS_SIMULATOR
void RTDS_DummyTraceFunction( void )
  {
#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
  RTDS_TraceAdd();
#endif
  }
#endif /* RTDS_SIMULATOR */
  
  
/*
** FUNCTION RTDS_SimulatorTrace:
** -----------------------------
** Handles trace with the host depending on the type of link available and options set in the profile.
**
** PARAMETERS:
**  event :           Trace event                
**  eventParameter1 : Parameter 1 of the trace   
**  eventParameter2 : Parameter 2 of the trace   
**  currentContext :  Calling context            
**  waitAck :         If true, an acknoledgement should be expected from the simulator we communicate with.
*/

#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
void RTDS_SimulatorTrace(
  enum RTDS_EventType       event,
  void                    * eventParameter1,
  long                      eventParameter2,
  RTDS_GlobalProcessInfo  * currentContext,
  int                       waitAck
)
  {
#if defined(RTDS_FORMAT_TRACE)
  unsigned long   sysTime = 0;
  int             formatStatus = RTDS_ERROR;
  char          * formattedCmd = NULL;
  size_t          command_size;
#endif /* defined(RTDS_FORMAT_TRACE) */

  /* Update global trace information */
  RTDS_globalTraceEntry.event = event;
  RTDS_globalTraceEntry.eventParameter1 = (void*)eventParameter1;
  RTDS_globalTraceEntry.eventParameter2 = (long)eventParameter2;
  RTDS_globalTraceEntry.currentContext = (RTDS_GlobalProcessInfo*)currentContext;

#if defined(RTDS_FORMAT_TRACE)
  formatStatus = RTDS_FormatTrace(RTDS_globalSystemTime, RTDS_globalTraceEntry, waitAck, &formattedCmd);
  if ( formattedCmd != NULL )
    {
#if defined(RTDS_SOCKET_PORT)
    /* Send Trace to the external program: RTDS or MSC tracer */
    command_size = strlen(formattedCmd);
    if ( send(globalClientSocketId, formattedCmd, command_size, 0) != command_size )
      {
      RTDS_SYSTEM_ERROR(RTDS_ERROR_SEND)
      }
    if ( waitAck == RTDS_DTRACE_ACK_WAIT )
      { 
      RTDS_DTRACE_ACKNOWLEDGE_WAIT; 
      }
#endif /* defined(RTDS_SOCKET_PORT) */
    }
#if defined(RTDS_SIMULATOR)
    RTDS_DummyTraceFunction();
#endif /* defined(RTDS_SIMULATOR) */
    RTDS_FREE( formattedCmd );
#elif defined(RTDS_SIMULATOR)
    if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
      {
      RTDS_messageDataToString(
        &RTDS_globalPrintableParameters,
        ((RTDS_MessageHeader*)eventParameter1)->messageNumber,
        ((RTDS_MessageHeader*)eventParameter1)->dataLength,
        (void *)(((RTDS_MessageHeader*)eventParameter1)->pData),
        RTDS_PARAM_CODEC_MAX_DEPTH
      );
      }
    RTDS_DummyTraceFunction();
    if ( ( event == RTDS_messageSent ) || ( event == RTDS_messageReceived ) )
        {
        RTDS_FREE(RTDS_globalPrintableParameters);
        RTDS_globalPrintableParameters = NULL;
        }
#endif /* defined(RTDS_FORMAT_TRACE) / defined(RTDS_SIMULATOR) */
  }
#endif /* defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACER) */


/*
** FUNCTION RTDS_GetMessageUniqueId:
** -------------------------------------
** Gets a message unique id for the simulator.
*/

#ifdef RTDS_SIMULATOR
unsigned long RTDS_GetMessageUniqueId( void )
  {
  unsigned char *index;
  long uniqueByteId;
  long uniqueBitId;

  index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;

  for ( uniqueByteId = 0 ; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE ; uniqueByteId++ )
    {
    if ( *index != 0xFF )
      {
      for ( uniqueBitId = 0 ; uniqueBitId < 8 ; uniqueBitId++ )
        {
        if ( ( ( 1 << uniqueBitId ) & *index ) == 0 )
          {
          *index = *index | ( 1 << uniqueBitId );
          RTDS_CRITICAL_SECTION_STOP;
          return (8 * uniqueByteId + uniqueBitId + 1);
          }
        }
      }
    index++;
    }

  /* All bits are set... No more message unique id */
  RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID )
  return 0;
  }
#endif

  
/*
** FUNCTION RTDS_ReleaseMessageUniqueId:
** -------------------------------------
** Make a message unique id available from the pool.
*/

#ifdef RTDS_SIMULATOR
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
  {
  unsigned char *index;

  if ( messageUniqueId == 0 ) /* probably a timer */
    return;
  messageUniqueId -= 1;
  index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;
  index += ( unsigned char )( messageUniqueId / 8 );
  ( *index ) = ( *index ) & ~( 1 << messageUniqueId % 8 );
  }
#endif


/*
** FUNCTION RTDS_IsSemaphoreAvailable:
** -----------------------------------
** Checks if a semaphore is available
*/

#if defined( RTDS_SIMULATOR ) || defined( RTDS_MSC_TRACER )
long RTDS_IsSemaphoreAvailable( RTDS_SemaphoreId semaphoreId )
  {
  /* Explicitely test for semaphore type and figure out its availability from its attributes */
  RTDS_Proc * semaphore_descriptor = (RTDS_Proc*)semaphoreId;
  switch(semaphore_descriptor->RTDS_currentContext->sdlProcessNumber)
    {
    case RTDS_internal_BinarySemaphoreProcess:
      return semaphore_descriptor->myLocals.RTDS_BinarySemaphoreProcess.is_available;
    case RTDS_internal_CountingSemaphoreProcess:
      return (semaphore_descriptor->myLocals.RTDS_CountingSemaphoreProcess.available_slots > 0);
    case RTDS_internal_MutexSemaphoreProcess:
      return (semaphore_descriptor->myLocals.RTDS_MutexSemaphoreProcess.take_count == 0);
    }
  return 0;
  }
#endif


/*
** FUNCTION RTDS_GetTimerUniqueId:
** -------------------------------
** Returns a new unique identifier for a timer.
*/

long RTDS_GetTimerUniqueId(void)
  {
  return 1;
  }


/*
** FUNCTION RTDS_StartTimer:
** -------------------------
** Starts a timer.
*/

#ifdef RTDS_HAS_TIMERS
void RTDS_StartTimer(RTDS_GlobalProcessInfo * context, long timerNumber, long timerUniqueId, int delay)
  {
  RTDS_TimerState * previousTimerDescriptor = NULL;
  RTDS_TimerState * timerDescriptor = NULL;
  RTDS_TimerState * startedTimerDescriptor = NULL;
  int               timerDelay = 0;
  
  /* We'll access the global list of timers: disable interrupts */
  RTDS_DISABLE_INTERRUPTS;
  /* Look for element in timer list just before the first active one */
  for ( previousTimerDescriptor = RTDS_globalTimerListFirst; previousTimerDescriptor != NULL; previousTimerDescriptor = previousTimerDescriptor->next )
    {
    if ( previousTimerDescriptor->next == RTDS_globalTimerListCurrent )
      break;
    }
  /* Look for the first timer in the list that is still active and that times-out after the one we start */
  for ( timerDescriptor = RTDS_globalTimerListCurrent; timerDescriptor != NULL; timerDescriptor = timerDescriptor->next )
    {
    /* If timer times out after the one we're starting, we've found its position */
    if ( timerDelay + timerDescriptor->timeoutDelay > delay )
      break;
    timerDelay += timerDescriptor->timeoutDelay;
    previousTimerDescriptor = timerDescriptor;
    }
  /* Insert timer at found position */
  startedTimerDescriptor = RTDS_NEW_TIMER_STATE;
  startedTimerDescriptor->timerNumber = timerNumber;
  startedTimerDescriptor->timerUniqueId = timerUniqueId;
  startedTimerDescriptor->timeoutValue = delay + RTDS_globalSystemTime;
  startedTimerDescriptor->timeoutDelay = delay - timerDelay;
  startedTimerDescriptor->receiverId = context->mySdlInstanceId;
  startedTimerDescriptor->state = RTDS_TIMER_OK;
  startedTimerDescriptor->next = timerDescriptor;
  if ( previousTimerDescriptor == NULL )
    {
    RTDS_globalTimerListFirst = startedTimerDescriptor;
    RTDS_globalTimerListCurrent = startedTimerDescriptor;
    }
  else
    {
    if ( previousTimerDescriptor->next == RTDS_globalTimerListCurrent )
      RTDS_globalTimerListCurrent = startedTimerDescriptor;
    previousTimerDescriptor->next = startedTimerDescriptor;
    }
  /* Update relative delay for next timer if any */
  if ( timerDescriptor != NULL )
    timerDescriptor->timeoutDelay -= startedTimerDescriptor->timeoutDelay;
  /* Enable back interrupts */
  RTDS_ENABLE_INTERRUPTS;
  }
#endif


/*
** FUNCTION RTDS_StopTimer:
** ------------------------
** Cancels a timer.
*/

#ifdef RTDS_HAS_TIMERS
void RTDS_StopTimer(RTDS_GlobalProcessInfo * context, long timerNumber)
  {
  RTDS_TimerState * timerDescriptor = NULL;
  
  /* Look for timer in list that has this timer number and is for this instance */
  for ( timerDescriptor = RTDS_globalTimerListFirst; timerDescriptor != NULL; timerDescriptor = timerDescriptor->next )
    {
    if ( ( timerDescriptor->timerNumber == timerNumber ) &&
         ( timerDescriptor->receiverId == context->mySdlInstanceId ) &&
         ( timerDescriptor->state == RTDS_TIMER_OK ) )
      {
      /* If found, just indicate the timer as cancelled in its descriptor */
      timerDescriptor->state = RTDS_TIMER_CANCELLED;
      return;
      }
    }
  /* If not found, cancel of a non-active timer: no big deal... */
  }
#endif


/*
** FUNCTION RTDS_ProcessForget:
** ----------------------------
** Forgets all information about a running process. Called when a process dies.
** Parameters:
** - RTDS_currentContext: context for the process to forget.
*/

void RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext )
  {
  RTDS_GlobalProcessInfo  * processInfo = NULL;
  RTDS_GlobalProcessInfo  * previousProcessInfo = NULL;
  RTDS_TimerState         * timer = NULL;
  RTDS_MessageHeader      * message = NULL;
  RTDS_MessageHeader      * tmpMessage = NULL;
  RTDS_SdlInstanceId      * instance2Delete = NULL;

  /* Free the current message if any */
  if ( RTDS_currentContext->currentMessage != NULL )
      {
#if defined( RTDS_SIMULATOR )
      /* Release the message unique id back to the pool */
      RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
#endif
      /* Free memory */
      RTDS_FREE_MESSAGE_HEADER( RTDS_currentContext->currentMessage );
      }

#ifdef RTDS_HAS_TIMERS
  /* Clean the timer chained list to free memory and stop watchdogs */
  for ( timer = RTDS_currentContext->timerList ; timer != NULL ; timer = RTDS_currentContext->timerList )
      {
      /* RTDS_TimerDelete( timer ); */
      RTDS_currentContext->timerList = timer->next;
      RTDS_FREE_TIMER_STATE( timer );
      }
#endif

  /* Clean the save queue: free messages and message unique ids in the read and write save queues */
  message = RTDS_currentContext->readSaveQueue;
  while( message != NULL )
      {
#if defined( RTDS_SIMULATOR )
      RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
#endif
      tmpMessage = message;
      message = message->next;
      RTDS_FREE_MESSAGE_HEADER( tmpMessage );
      }

  message = RTDS_currentContext->writeSaveQueue;
  while( message != NULL )
      {
#if defined( RTDS_SIMULATOR )
      RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
#endif
      tmpMessage = message;
      message = message->next;
      RTDS_FREE_MESSAGE_HEADER( tmpMessage );
      }

  /* Forget and free the process information in the global instance chained list */
  previousProcessInfo = NULL;
  for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
      {
      /* If instance was found */
      if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )
          {
          RTDS_SIMULATOR_TRACE( RTDS_processDied , processInfo , NULL , RTDS_currentContext );
          /* Remove instance descriptor from list */
          if ( previousProcessInfo == NULL )
              RTDS_globalProcessInfo = processInfo->next;
          else
              previousProcessInfo->next = processInfo->next;

          /* Free the process info block */
          RTDS_FREE_INSTANCE_ID( processInfo->mySdlInstanceId );
          RTDS_FREE_INSTANCE_INFO( processInfo );
          break;
          }
      previousProcessInfo = processInfo;
      }
  }


/*
** FUNCTION RTDS_TimerCleanUp:
** ---------------------------
** Called when the last received message is a timer to figure out if
** timer was not cancelled before it was received.
** Parameters:
** - currentContext: context of the current task
*/

void RTDS_TimerCleanUp(RTDS_GlobalProcessInfo * currentContext)
  {
#ifdef RTDS_HAS_TIMERS
  RTDS_TimerState * timer = NULL;
  RTDS_TimerState * prevTimer = NULL;

  prevTimer = NULL;
  for ( timer = currentContext->timerList ; timer != NULL; timer = timer->next )
    {
    /* If timer found and cancelled */
    if ( timer->timerUniqueId == currentContext->currentMessage->timerUniqueId )
      {
      if ( timer->state == RTDS_TIMER_CANCELLED )
        {
        /* Discard current message */
        RTDS_FREE_MESSAGE_HEADER( currentContext->currentMessage );
        currentContext->currentMessage = NULL;
        }
      /* Remove it from list of timers */
      if ( prevTimer == NULL )
        currentContext->timerList = currentContext->timerList->next;
      else
        prevTimer->next = timer->next;
      RTDS_FREE_TIMER_STATE( timer );
      break;
      }
    prevTimer = timer;
    }
#endif
  }


/*
** FUNCTION RTDS_SemaphoreRegister:
** --------------------------------
** Registers a newly created semaphore in the global semaphore chained list.
** Parameters:
**  - parentScheduler: Parent scheduler for the current instance.
**  - semaphoreNumber: Identifier for the semaphore name.
**  - semaphoreId: Identifier for the semaphore itself.
**  - creatorContext: Context for the creator instance.
*/

#ifdef RTDS_HAS_SEMAPHORES
void RTDS_SemaphoreRegister(RTDS_Scheduler * parentScheduler, int semaphoreNumber, RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * creatorContext)
  {
  long                        semIsAvailable = -1;
  RTDS_GlobalSemaphoreInfo  * new_semaphore_info;

  /* Register semaphore pseudo-process in parent scheduler */
  /* RTDS_Scheduler_registerInstance(parentScheduler, semaphoreId, 0); */
  /* Allocate and fill in a new semaphore info structure */
  new_semaphore_info = RTDS_NEW_SEMAPHORE_INFO;

  new_semaphore_info->semaphoreId = semaphoreId;
  new_semaphore_info->semaphoreNumber = semaphoreNumber;

  /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
  RTDS_CRITICAL_SECTION_START;
  new_semaphore_info->next = RTDS_globalSemaphoreInfo;
  RTDS_globalSemaphoreInfo = new_semaphore_info;
  RTDS_CRITICAL_SECTION_STOP;

#if defined(RTDS_SIMULATOR) || defined(RTDS_MSC_TRACER)
  semIsAvailable = RTDS_IsSemaphoreAvailable(semaphoreId);
#endif
  RTDS_SIMULATOR_TRACE(RTDS_semaphoreCreated, semaphoreId, semIsAvailable, creatorContext);
  }
#endif


/*
** FUNCTION RTDS_BinarySemaphoreCreate:
** ------------------------------------
** Creates a new binary semaphore.
** Parameters:
**  - parentScheduler: Parent scheduler for the current instance.
**  - initialState: If 0, semaphore is created empty.
**  Returns the created semaphore.
*/

#ifdef RTDS_HAS_SEMAPHORES
RTDS_SemaphoreId RTDS_BinarySemaphoreCreate(RTDS_Scheduler * parentScheduler, int initialState)
  {
  RTDS_Proc * RTDS_new_binary_semaphore;
  
  RTDS_new_binary_semaphore = RTDS_NEW_SEMAPHORE_PROC;
  RTDS_Proc_RTDS_BinarySemaphoreProcess_createInstance(RTDS_new_binary_semaphore, parentScheduler, NULL, NULL);
  RTDS_new_binary_semaphore->myLocals.RTDS_BinarySemaphoreProcess.is_available = initialState;
  
  return (RTDS_SemaphoreId)RTDS_new_binary_semaphore;
  }
#endif


/*
** FUNCTION RTDS_CountingSemaphoreCreate:
** --------------------------------------
** Creates a new counting semaphore.
** Parameters:
**  - parentScheduler: Parent scheduler for the current instance.
**  - initialCount: Initial number of available slots in semaphore.
** Returns the created semaphore.
*/

#ifdef RTDS_HAS_SEMAPHORES
RTDS_SemaphoreId RTDS_CountingSemaphoreCreate(RTDS_Scheduler * parentScheduler, int initialCount)
  {
  RTDS_Proc * RTDS_new_counting_semaphore;
  
  RTDS_new_counting_semaphore = RTDS_NEW_SEMAPHORE_PROC;
  RTDS_Proc_RTDS_CountingSemaphoreProcess_createInstance(RTDS_new_counting_semaphore, parentScheduler, NULL, NULL);
  RTDS_new_counting_semaphore->myLocals.RTDS_CountingSemaphoreProcess.available_slots = initialCount;
  
  return (RTDS_SemaphoreId)RTDS_new_counting_semaphore;
  }
#endif


/*
** FUNCTION RTDS_MutexSemaphoreCreate:
** -----------------------------------
** Creates a new mutex semaphore.
** Parameters:
**  - parentScheduler: Parent scheduler for the current instance.
** Returns the created semaphore.
*/

#ifdef RTDS_HAS_SEMAPHORES
RTDS_SemaphoreId RTDS_MutexSemaphoreCreate(RTDS_Scheduler * parentScheduler)
  {
  RTDS_Proc * RTDS_new_mutex_semaphore;
  
  RTDS_new_mutex_semaphore = RTDS_NEW_SEMAPHORE_PROC;
  RTDS_Proc_RTDS_MutexSemaphoreProcess_createInstance(RTDS_new_mutex_semaphore, parentScheduler, NULL, NULL);
  return (RTDS_SemaphoreId)RTDS_new_mutex_semaphore;
  }
#endif


/*
** FUNCTION RTDS_GetSemaphoreId:
** -----------------------------
** Returns an actual semaphore descriptor corresponding to a semaphore name.
** Parameters:
**  - semaphoreNumber: Numerical identifier for the semaphore name.
** Returns the instance descriptor for the found semaphore.
*/

#ifdef RTDS_HAS_SEMAPHORES
RTDS_SdlInstanceId * RTDS_GetSemaphoreId(int semaphoreNumber)
  {
  RTDS_GlobalSemaphoreInfo  * semaphore_info;
  RTDS_SemaphoreId            found_semaphore = NULL;

  RTDS_CRITICAL_SECTION_START;
  for ( semaphore_info = RTDS_globalSemaphoreInfo ; semaphore_info != NULL ; semaphore_info = semaphore_info->next )
    {
    if ( semaphore_info->semaphoreNumber == semaphoreNumber )
      {
      found_semaphore = semaphore_info->semaphoreId;
      break;
      }
    }
  RTDS_CRITICAL_SECTION_STOP;

  if ( found_semaphore == NULL )
    {
    RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_SEMAPHORE_ID )
    }
  return ((RTDS_Proc*)found_semaphore)->RTDS_currentContext->mySdlInstanceId;
  }
#endif

