/* Include unistd when using a unix based system or cygwin */
#ifdef __CYGWIN__
#include <unistd.h>
#else
#include <stdlib.h>
#endif

#include "RTDS_Common.h"
#include "RTDS_SemaphoreProcess.h"

#include "RTDS_MACRO.h"


/*
** Macros and function for waiting instance descriptors allocation:
** ----------------------------------------------------------------
*/

/* If there's a configured maximum number of waiting instances */
#ifdef RTDS_MAX_WAITING_INSTANCES
/* Waiting instance allocation is static: define static array and functions simulating a malloc & a free */
static char RTDS_waiting_instances_initialized = 0;
static RTDS_WaitingProcessList RTDS_waiting_instance_array[RTDS_MAX_WAITING_INSTANCES];
static char RTDS_waiting_instance_used[RTDS_MAX_WAITING_INSTANCES];

static RTDS_WaitingProcessList * RTDS_allocate_waiting_instance(void)
  {
  int i;
  
  if ( ! RTDS_waiting_instances_initialized )
    {
    for ( i = 0; i < RTDS_MAX_WAITING_INSTANCES; i++ ) RTDS_waiting_instance_used[i] = 0;
    RTDS_waiting_instances_initialized = 1;
    }
  for ( i = 0; i < RTDS_MAX_WAITING_INSTANCES; i++ )
    {
    if ( ! RTDS_waiting_instance_used[i] )
      {
      RTDS_waiting_instance_used[i] = 1;
      return & RTDS_waiting_instance_array[i];
      }
    }
  return NULL;
  }

static void RTDS_free_waiting_instance(RTDS_WaitingProcessList * waiting_instance)
  {
  int i;
  for ( i = 0; i < RTDS_MAX_WAITING_INSTANCES; i++ )
    {
    if ( & RTDS_waiting_instance_array[i] == waiting_instance )
      {
      RTDS_waiting_instance_used[i] = 0;
      return;
      }
    }
  RTDS_SYSTEM_ERROR(4);
  }
  
#define RTDS_NEW_WAITING_INSTANCE RTDS_allocate_waiting_instance()
#define RTDS_FREE_WAITING_INSTANCE(WAITING_INSTANCE) RTDS_free_waiting_instance(WAITING_INSTANCE)

/* If there is no configured maximum number of instances */
#else

/* If dynamic memory allocation is allowed */
#ifdef RTDS_MALLOC
/* Creating a timer dynamically allocates a descriptor */
#define RTDS_NEW_WAITING_INSTANCE (RTDS_WaitingProcessList *)RTDS_MALLOC(sizeof(RTDS_WaitingProcessList))
#define RTDS_FREE_WAITING_INSTANCE(WAITING_INSTANCE) RTDS_FREE(WAITING_INSTANCE)
/* If dynamic memory allocation is not allowed, it's an error */
#else
#error ERROR! Dynamic memory allocation is not available (no RTDS_MALLOC macro), but no maximum number of instances defined via the RTDS_MAX_WAITING_INSTANCES macro!
#endif

#endif



/*
** FUNCTION RTDS_SemaphoreProcess_init:
** ------------------------------------
** Initialization function for common semaphore attributes
*/
void RTDS_SemaphoreProcess_init(RTDS_SemaphoreProcessCommon * semaphore_attributes)
  {
  semaphore_attributes->waiting_processes_head = NULL;
  semaphore_attributes->waiting_processes_last = NULL;
  }


/*
** FUNCTION RTDS_SemaphoreProcess_addWaitingInstance:
** --------------------------------------------------
** Adds an instance to the list of those waiting for the semaphore
*/

void RTDS_SemaphoreProcess_addWaitingInstance(RTDS_SemaphoreProcessCommon * semaphore_attributes, RTDS_SdlInstanceId * instance_id)
  {
  if ( semaphore_attributes->waiting_processes_last == NULL )
    {
    semaphore_attributes->waiting_processes_head = RTDS_NEW_WAITING_INSTANCE;
    semaphore_attributes->waiting_processes_last = semaphore_attributes->waiting_processes_head;
    }
  else
    {
    semaphore_attributes->waiting_processes_last->next = RTDS_NEW_WAITING_INSTANCE;
    semaphore_attributes->waiting_processes_last = semaphore_attributes->waiting_processes_last->next;
    }
  semaphore_attributes->waiting_processes_last->sdlInstanceId = instance_id;
  semaphore_attributes->waiting_processes_last->next = NULL;
  }


/*
** FUNCTION RTDS_SemaphoreProcess_popFirstWaitingInstance:
** -------------------------------------------------------
** Pops the first waiting instance id in the list and returns it in returned_id.
** Returns 1 if there are waiting instances, and 0 if there aren't
*/

short RTDS_SemaphoreProcess_popFirstWaitingInstance(RTDS_SemaphoreProcessCommon * semaphore_attributes, RTDS_SdlInstanceId ** returned_id)
  {
  RTDS_WaitingProcessList * waiting_process_current;
  
  if ( semaphore_attributes->waiting_processes_head == NULL )
    return 0;
  *returned_id = semaphore_attributes->waiting_processes_head->sdlInstanceId;
  waiting_process_current = semaphore_attributes->waiting_processes_head;
  semaphore_attributes->waiting_processes_head = semaphore_attributes->waiting_processes_head->next;
  if ( semaphore_attributes->waiting_processes_head == NULL )
    semaphore_attributes->waiting_processes_last = NULL;
  RTDS_FREE_WAITING_INSTANCE(waiting_process_current);
  return 1;
  }


/*
** FUNCTION RTDS_SemaphoreProcess_removeWaitingInstance:
** -----------------------------------------------------
** Removes an instance to the list of those waiting for the semaphore. Returns 1
** if the instance was found in the waiting ones, 0 if it wasn't.
*/

short RTDS_SemaphoreProcess_removeWaitingInstance(RTDS_SemaphoreProcessCommon * semaphore_attributes, RTDS_SdlInstanceId * instance_id)
  {
  RTDS_WaitingProcessList * waiting_process_previous = NULL;
  RTDS_WaitingProcessList * waiting_process_current;
  
  /* Browse list of waiting instances */
  for ( waiting_process_current = semaphore_attributes->waiting_processes_head;
        waiting_process_current != NULL;
        waiting_process_current = waiting_process_current->next )
    {
    /* If given instance found, remove it from the list */
    if ( waiting_process_current->sdlInstanceId == instance_id )
      {
      if ( waiting_process_previous == NULL )
        semaphore_attributes->waiting_processes_head = semaphore_attributes->waiting_processes_head->next;
      else
        waiting_process_previous->next = waiting_process_current->next;
      if ( waiting_process_current == semaphore_attributes->waiting_processes_last )
        semaphore_attributes->waiting_processes_last = waiting_process_previous;
      RTDS_FREE_WAITING_INSTANCE(waiting_process_current);
      return 1;
      }
    }
  /* If we get there, instance not found */
  return 0;
  }

