#ifndef _RTDS_BASIC_TYPES_H_
#define _RTDS_BASIC_TYPES_H_

/* PragmaDev RTDS FreeRTOS integration */

#ifdef RTDS_FREERTOS_WINDOWS_SIMULATOR
    #include <winsock2.h>
    #include <windows.h>

    #define RTDS_THREAD_START_FUNC LPTHREAD_START_ROUTINE
    #define RTDS_DEFAULT_TCP_CLIENT_PRIORITY    THREAD_PRIORITY_TIME_CRITICAL
#endif

/**********************************
/* FreeRTOS include */
#include "FreeRTOS.h"
#include "queue.h"
#include "semphr.h"
#include "task.h"
#include "timers.h"


#define RTDS_TIMER_FINISHED     2
#define RTDS_TIMER_OK           1
#define RTDS_TIMER_CANCELLED    0

/*
 * DEFINE FOR PROCESS CREATION:
 * ----------------------------
 */
#define RTDS_TASK_OPTIONS                   0       		                    /* Not used with FreeRTOS */
#define RTDS_TASK_STACK_SIZE                4096                          /* Stack size when creating a new task */
#define RTDS_DEFAULT_PROCESS_PRIORITY       3 	                           /* If priority not defined in SDL-RT, it will be given this value */


/*
 * DEFINE FOR QUEUE MANIPULATION:
 * ------------------------------
 */
#define RTDS_QUEUE_MAX_MSG                  20		   			                    /* Maximum number of messages in a queue */
#define RTDS_QUEUE_MAX_MSG_LENGTH           sizeof(RTDS_MessageHeader *)	 /* Maximum length of messages in a queue */
#define RTDS_WAIT_FOREVER                   0                        					/* Mode to wait forever on a queue */


/*
 * DEFINE FOR SEMAPHORE MANIPULATION:
 * ------------------------------
 */
#define RTDS_SEMAPHORE_TIME_OUT_FOREVER     portMAX_DELAY    			           /* Time out option when taking the semaphore : wait forever */
#define RTDS_SEMAPHORE_TIME_OUT_NO_WAIT     0 		    				                   /* Time out option when taking the semaphore : no wait */

#define RTDS_SEMAPHORE_OPTION_PRIO		        0  					                       /* Ignored when using FreeRTOS */
#define RTDS_SEMAPHORE_OPTION_FIFO		        0						                        /* Ignored when using FreeRTOS */

#define RTDS_BINARY_SEM_INITIAL_EMPTY       0 							                      /* Binary semaphore creation initial state: empty */
#define RTDS_BINARY_SEM_INITIAL_FULL	       1					                         /* Binary semaphore creation initial state: full */
#define RTDS_BINARY_MAX_COUNT               127                            /* Maximum value for binary semaphore */

#define RTDS_MUTEX_SEM_DELETE_SAFE		        0				                          /* Ignored when using FreeRTOS */
#define RTDS_MUTEX_SEM_INVERSION_SAFE       0                       							/* Ignored when using FreeRTOS */



/* ************************************************************************************ *
 *									COVERAGE INFORMATION
 * ************************************************************************************ *
 * The coverage information is stored in an array of unsigned char. The SDL-RT symbol
 * number is used as an index; the value stored is the number of times the symbol has
 * been executed. It can not exceed 0xFF for each symbol.
 * ************************************************************************************ */
#ifdef RTDS_COVERAGE_NB_SYMBOLS
#define RTDS_COVERAGE_DECL		unsigned char  RTDS_globalCoverage[RTDS_COVERAGE_NB_SYMBOLS];
#define RTDS_COVERAGE_PROTO 	extern RTDS_COVERAGE_DECL
#define RTDS_COVERAGE_INIT \
	{ \
	int RTDS_i; \
	for (RTDS_i=0;RTDS_i<RTDS_COVERAGE_NB_SYMBOLS;RTDS_i++) \
		RTDS_globalCoverage[RTDS_i]=0; \
	}
#define RTDS_COVERAGE_LOG(SYMBOL_NUMBER) \
	if (RTDS_globalCoverage[SYMBOL_NUMBER]!=0xFF) RTDS_globalCoverage[SYMBOL_NUMBER]++;
#else
#define RTDS_COVERAGE_DECL
#define RTDS_COVERAGE_PROTO
#define RTDS_COVERAGE_INIT
#endif



/*
 * DEFINE FOR SIMULATION:
 * ------------------------------
 * Define on how many bytes the messageUniqueId pool will be made of
 * 1 means 1 byte means 1*8 bits means 8 possible simultaneous values
 * Maximum is 8192 since the id a long type
 */
#ifdef RTDS_SIMULATOR
#define RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE	    64
#define RTDS_PARAM_CODEC_MAX_DEPTH				4 	/* Number of levels when formatting message parameters to be printable in the SDL-RT debugger */
#define RTDS_PARAM_CODEC_CHUNK_SIZE 			128 /* Min size of memory chunk used when formatting message parameters to be printable in the SDL-RT debugger */
#endif


/*
 * CRITICAL SECTION DECLARATIONS AND MACROS:
 * ----------------------------------------
 */
#define RTDS_CRITICAL_SECTION_DECL		RTDS_SemaphoreId RTDS_globalSystemSemId=NULL;
#define RTDS_CRITICAL_SECTION_PROTO 	extern RTDS_SemaphoreId RTDS_globalSystemSemId;
#define RTDS_CRITICAL_SECTION_INIT \
    RTDS_globalSystemSemId = xSemaphoreCreateMutex(); \
	if (RTDS_globalSystemSemId == NULL) \
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEMAPHORE_CREATE);
#define RTDS_CRITICAL_SECTION_START 	xSemaphoreTake(RTDS_globalSystemSemId,portMAX_DELAY);
#define RTDS_CRITICAL_SECTION_STOP		xSemaphoreGive(RTDS_globalSystemSemId);
#define RTDS_CRITICAL_SECTION_POSTAMBLE 
/* If INCLUDE_vTaskSuspend is set to '1' then specifying the block time as portMAX_DELAY will cause the task to block indefinitely (without a timeout). */

/*
 * STARTUP SYNCHRONISATION DECLARATIONS AND MACROS:
 * -----------------------------------------------
 */
#define RTDS_HOLD 									1 /* The newly created task will held at startup on the startup semaphore */
#define RTDS_NO_HOLD								0 /* The newly created task will not held at startup on the startup semaphore */
#define RTDS_START_SYNCHRO_DECL 		RTDS_SemaphoreId RTDS_globalStartSynchro;
#define RTDS_START_SYNCHRO_PROTO		extern RTDS_SemaphoreId RTDS_globalStartSynchro;
#define RTDS_START_SYNCHRO_INIT \
	RTDS_globalStartSynchro  = xSemaphoreCreateBinary(); \
    if (RTDS_globalStartSynchro == NULL) \
		RTDS_SYSTEM_ERROR(RTDS_ERROR_SEMAPHORE_CREATE);
#define RTDS_START_SYNCHRO_WAIT \
	xSemaphoreTake( RTDS_globalStartSynchro, portMAX_DELAY); \
	xSemaphoreGive( RTDS_globalStartSynchro );
#define RTDS_START_SYNCHRO_GO \
	xSemaphoreGive( RTDS_globalStartSynchro );
#define RTDS_START_SYNCHRO_POSTAMBLE	xSemaphoreGive( RTDS_globalStartSynchro );

#ifdef RTDS_SIMULATOR
#define RTDS_START_SYNCHRO_HOLD \
	xSemaphoreTake( RTDS_globalStartSynchro, portMAX_DELAY);
#define RTDS_START_SYNCHRO_UNHOLD \
	xSemaphoreGive( RTDS_globalStartSynchro );
#else
#define RTDS_START_SYNCHRO_HOLD ;
#define RTDS_START_SYNCHRO_UNHOLD ;
#endif



/*
 * TYPE RTDS_QUEUE.ID:
 * ----------------------
 * Type for a message queue identifier
 */
typedef xQueueHandle RTDS_RtosQueueId;

/*
 * TYPE RTDS_SEMAPHORE.ID:
 * ----------------------
 * Type for a semaphore identifier
 */
typedef SemaphoreHandle_t RTDS_SemaphoreId;

/*
 * TYPE RTDS_SEMAPHORE_STATUS:
 * ----------------------
 * Type for a semaphore status
 */
typedef portBASE_TYPE RTDS_SemaphoreStatus;

#define RTDS_OK 		pdTRUE
#define RTDS_ERROR	    pdFAIL

/*
 * TYPE RTDS_SEMAPHORE_TIMEOUT:
 * ----------------------
 * Type for a semaphore timeout value
 */
typedef portTickType RTDS_SemaphoreTimeout;

/*
 * TYPE RTDS_PROCESS_ID:
 * ----------------------
 * Type for a pid
 */

typedef xTaskHandle RTDS_RtosTaskId;

/*
 * TYPE RTDS_PROCESS_PRIORITY:
 * ----------------------
 * Type for process priority
 */
typedef unsigned portBASE_TYPE RTDS_ProcessPriority;


/*
 * STRUCT RTDS_TIMER.STATE:
 * ------------------------
 * Element in chained list for all active timers
 */
#define RTDS_TIMER_OK           1
#define RTDS_TIMER_CANCELLED    0

typedef struct RTDS_TimerState
    {
    long                        timerNumber;                /* Message number for the timer name                            */
    long                        timerUniqueId;              /* Timer's unique identifier among all timers                   */
    unsigned long               timeoutValue;               /* System tick counter value when it will go off                */
    struct RTDS_SdlInstanceId   * receiverSdlInstanceId;    /* The message queue of the receiver                            */
    int                         state;                      /* State for the timer (RTDS_TIMER_OK or RTDS_TIMER_CANCELLED)  */
    xTimerHandle                watchDogId; 			             /* Id of the watchDog i.e. of the FreeRTOS cyclic timer              */
    struct RTDS_TimerState      * next;	                    /* Next timer in list                                           */
    } RTDS_TimerState;



/*
 * STRUCT RTDS_GLOBAL.SEMAPHORE.INFO:
 * ----------------------------------
 * Type for the list of semaphores in the system with their information
 */
typedef struct RTDS_GlobalSemaphoreInfo
	{
	RTDS_SemaphoreId                    semaphoreId;        /* The semaphore address */
	int                                 semaphoreNumber;    /* Semaphore number representing its name */
	struct RTDS_GlobalSemaphoreInfo     *next;              /* Next semaphoreInfo */
	} RTDS_GlobalSemaphoreInfo;


#ifdef RTDS_SIMULATOR
/* Global variable pointing to decoded parmaters */
extern char *RTDS_globalPrintableParameters;
#endif

/*
 * Additional fields in common types
 * ---------------------------------
 */
/* Message header: none */
#define RTDS_MESSAGE_HEADER_ADDITIONNAL_FIELDS
/* Process information: add priority */
#define RTDS_GLOBAL_PROCESS_INFO_ADDITIONNAL_FIELDS \
  RTDS_ProcessPriority priority;

#endif /* _RTDS_BASIC_TYPES_H_ */
