"""
This module implements the server-side for the LLDB interface for PragmaDev Studio.
It is intended to be run by the embedded Python interpreter in the lldb debugger,
with a command such as:
script import lldb_command_interpreter; lldb_command_interpreter.run()

See file lldb_commands.txt for the actual lldb commands used to configure and run
the interface.

The basic principle is to have a socket server within the debugger that gets commands
from the PragmaDev Studio debugger and sends the answers back via the socket. Commands
and answers are strings with parts separated by a tab and ending with a line feed.
"""

import sys
import codecs
import os
import socket
import threading
import lldb

## Socket connected to the client. There can be only one client.
client_socket = None

## Lock for the socket, to avoid having several threads trying to send something into it at the same time
client_socket_lock = threading.Lock()

## Name for the debugged program executable. Used when restarting the debugging session.
program_executable = None

## Current target in the debugger (lldb.SBTarget)
current_target = None

## Current process in the debugger (lldb.SBProcess)
current_process = None

## Current thread in the debugger (lldb.SBThread)
current_thread = None

## List of breakpoints (lldb.SBBreakpoint). The index in this list is the breakpoint id.
breakpoints = []

## (Set to True to debug the interface)
DEBUG = False

## Conversion functions between bytes & strings, needed for Python 3
B = (lambda s: bytes(s, 'utf-8')) if sys.version_info[0] >= 3 else lambda s: s
S = (lambda b: str(b, 'utf-8')) if sys.version_info[0] >= 3 else lambda s: s


## FUNCTION load_program:
## ----------------------

def load_program(program_executable_to_load):
  """
  Loads the program in lldb. Called by the socket command 'ldpgm'.
  
  @param program_executable_to_load: File name for the program executable to load.
  @type program_executable_to_load: string
  """
  global program_executable
  global current_target
  ## Remember file name for program executable
  program_executable = program_executable_to_load
  ## Actually load program in the debugger to create the target
  current_target = lldb.debugger.CreateTarget(program_executable)
  return None


## FUNCTION stop_callback:
## -----------------------

def stop_callback(frame=None, *args):
  """
  Function called whenever the program execution stops. It is set as an explicit callback
  for breakpoints, is explicitely called when a 'stop' command is issued and is called
  automatically whenever the program stops.
  
  Note that this means it can actually be called several times when the program stops, so
  this case has to be handled in the client (PragmaDev Studio debugger).
  
  @param frame: Execution frame where the program stopped, if known. This is passed automatically
                for a breakpoint, but not in other cases.
  @type frame: lldb.SBFrame
  
  @return: Nothing. The information about where the execution stopped is sent directly through
           the socket.
  """
  global current_thread
  ## If no frame passed
  if frame is None:
    ## If we don't know the current frame, use the global variable in lldb (which appears to be often wrong, but we don't have a choice here...)
    if current_thread is None:
      frame = lldb.frame
    ## If we know the current frame, use this one
    else:
      frame = current_thread.GetSelectedFrame()
  ## In all cases, get the current thread from the current frame
  current_thread = frame.thread
  ## Build information to send back to the client: break address, function name, file name, line number, thread id & thread name
  stop_info = 'STOP\t%s\t%s\t%s\t%s\t%s\t%s\n' % (frame.pc, frame.GetFunctionName(), frame.GetLineEntry().file, frame.GetLineEntry().line, current_thread.id, current_thread.name)
  if DEBUG:
    print('Stop %s' % repr(stop_info))
  ## Send information to the client through the socket
  try:
    client_socket_lock.acquire()
    client_socket.send(B(stop_info))
    client_socket_lock.release()
  ## Client socket might be dead if the debugger has closed before the execution actually ended: no big deal
  except socket.error:
    pass
  

## FUNCTION _configure_breakpoint:
## -------------------------------

def _configure_breakpoint(breakpoint, nb_hits, volatile):
  """
  Configures a newly created breakpoint.
  
  @param breakpoint: Breakpoint to configure.
  @type breakpoint: lldb.SBBreakpoint
  
  @param nb_hits: Number of hits before the breakpoint actually fires.
  @type nb_hits: integer as a string
  
  @param volatile: If true, the breakpoint will self-destruct the first time it is hit.
  @type volatile: one-character string
  
  @return: The infrmation to send back to the cleint through the socket: breakpoint id,
           file name and line number.
  @rtype: string
  """
  ## Configure ignore count
  if nb_hits != '-':
    breakpoint.SetIgnoreCount(int(nb_hits))
  ## Configure "one shot"
  if volatile == 'Y':
    breakpoint.SetOneShot(True)
  ## Set triggered callback when the breakpoint is hit to our stop callback
  breakpoint.SetScriptCallbackFunction('lldb_command_interpreter.stop_callback')
  ## Record breakpoint and remember its identifier
  breakpoint_id = len(breakpoints)
  breakpoints.append(breakpoint)
  ## Search for file name and line number where the breakpoint was set
  file_name, line_number = '-', 0
  for i in range(breakpoint.num_locations):
    breakpoint_line_entry = breakpoint.GetLocationAtIndex(i).GetAddress().line_entry
    ## A line set to 0 means the location is not known; ignore it
    if breakpoint_line_entry.line > 0:
      file_name, line_number = breakpoint_line_entry.file.fullpath, breakpoint_line_entry.line
      break
  ## Return information to send back through the socket
  return '%s\t%s\t%s' % (breakpoint_id, file_name, line_number)


## FUNCTION set_breakpoint_on_function:
## ------------------------------------

def set_breakpoint_on_function(function_name, nb_hits, volatile):
  """
  Sets a breakpoint on a function identified by its name. Called by the 'sbpfunc' command
  received via the socket.
  
  @param function_name: Name of the function on which the breakpont must be set.
  @type function_name: string
  
  @param nb_hits: Number of hits before the breakpoint actually fires.
  @type nb_hits: integer as a string
  
  @param volatile: If true, the breakpoint will self-destruct the first time it is hit.
  @type volatile: one-character string
  
  @return: The infrmation to send back to the cleint through the socket: breakpoint id,
           file name and line number.
  @rtype: string
  """
  ## Create breakpoint
  breakpoint = current_target.BreakpointCreateByName(function_name)
  ## Configure breakpoint and return the information on it
  return _configure_breakpoint(breakpoint, nb_hits, volatile)
  

## FUNCTION set_breakpoint_on_file:
## --------------------------------

def set_breakpoint_on_file(file_name, line_number, nb_hits, volatile):
  """
  Sets a breakpoint on a line number in a file. Called by the 'sbpfile' command received
  through the socket.
  
  @param file_name: Name of the file on which the breakpoint must be set.
  @type file_name: string
  
  @param line_number: Line number in the file where the breakpoint must be set.
  @type line_number: integer as a string
  
  @param nb_hits: Number of hits before the breakpoint actually fires.
  @type nb_hits: integer as a string
  
  @param volatile: If true, the breakpoint will self-destruct the first time it is hit.
  @type volatile: one-character string
  
  @return: The infrmation to send back to the cleint through the socket: breakpoint id,
           file name and line number.
  @rtype: string
  """
  ## Create breakpoint
  breakpoint = current_target.BreakpointCreateByLocation(file_name, int(line_number))
  ## Configure breakpoint and return the information on it
  return _configure_breakpoint(breakpoint, nb_hits, volatile)
  

## FUNCTION delete_breakpoint:
## ---------------------------

def delete_breakpoint(breakpoint_id):
  """
  Deletes a breakpoint identified by its index in the breakpoints list. Called by the
  'delbp' command received through the socket.
  
  @param breakpoint_id: Index of the breakpoint to delete.
  @type breakpoint_id: integer as a string
  """
  ## Get breakpoint and delete it
  current_target.BreakpointDelete(breakpoints[int(breakpoint_id)].GetID())
  return None


## FUNCTION run_program:
## ---------------------

def run_program(function_name):
  """
  Runs the program, optionally from a given function. Called by the 'run' command received
  through the socket.
  
  @param function_name: Name for the function to call if any.
  @type function_name: string
  """
  global current_process
  global current_thread
  ## If no function name given, just run the deugged program
  if function_name == '-':
    ## NB: running the program is asynchronous by default, so this will not wait for the program
    ## to stop and return immediately
    current_process = current_target.LaunchSimple([], None, os.getcwd())
  ## (The case where the function is specified is not supported yet; needed: ???)
  ## The program is multi-threaded and run as a whole, so there is no current thread yet
  current_thread = None
  return None


## FUNCTION continue_program:
## --------------------------

def continue_program():
  """
  Continues execution after a break. Called by the 'cont' command received through the
  socket.
  """
  global current_thread
  ## Current thread is no more known
  current_thread = None
  ## Continue execution
  current_process.Continue()
  return None


## FUNCTION step_over:
## -------------------

def step_over():
  """
  Executes one line in the current thread, not entering into any function called by this
  line. Called by the 'step' command received through the socket.
  """
  global current_thread
  ## For some reason, when doing an asynchronous step, the stop callback set via the stop
  ## hook doesn't seem to get called => do a synchronous one and call the callback ourselves
  lldb.debugger.SetAsync(False)
  ## Remember current thread as we will step in it
  current_thread = current_process.GetSelectedThread()
  ## Do one step over
  current_thread.StepOver(lldb.eAllThreads)
  ## Back to asynchronous mode
  lldb.debugger.SetAsync(True)
  ## Call stop callback
  stop_callback()
  return None


## FUNCTION step_in:
## -----------------

def step_in():
  """
  Executes one step of execution, entering any function that the current line calls.
  Called by the 'stepin' command received through the socket.
  """
  global current_thread
  ## For some reason, when doing an asynchronous step, the stop callback set via the stop
  ## hook doesn't seem to get called => do a synchronous one and call the callback ourselves
  lldb.debugger.SetAsync(False)
  ## Remember current thread as we will step in it
  current_thread = current_process.GetSelectedThread()
  ## Do one step into
  current_thread.StepInto(lldb.eAllThreads)
  ## Back to asynchronous mode
  lldb.debugger.SetAsync(True)
  ## Call stop callback
  stop_callback()
  return None


## FUNCTION step_out:
## ------------------

def step_out():
  """
  Runs execution until it goes out of the current execution frame. Called by the 'stepout'
  command received through the socket.
  """
  global current_thread
  ## For some reason, when doing an asynchronous step, the stop callback set via the stop
  ## hook doesn't seem to get called => do a synchronous one and call the callback ourselves
  lldb.debugger.SetAsync(False)
  ## Remember current thread as we will step in it
  current_thread = current_process.GetSelectedThread()
  ## Do one step out
  current_thread.StepOut()
  ## Back to asynchronous mode
  lldb.debugger.SetAsync(True)
  ## Call stop callback
  stop_callback()
  return None


## FUNCTION stop_program:
## ----------------------

def stop_program():
  """
  Stops the running program. Called by the 'stop' command received through the socket.
  """
  ## Stop the running program
  current_process.Stop()
  ## Explicitely call the stop callback, as we seem to have much more information in this
  ## context than when the callback is called automatically by the target stop hook
  stop_callback()
  return None


## FUNCTION get_active_thread_info:
## --------------------------------

def get_active_thread_info():
  """
  Returns the information about the current thread: thread identifier & function name.
  Called by the 'getth' command received through the socket.
  
  @rtype: string
  """
  ## Get current thread
  if current_thread is None:
    th = current_process.GetSelectedThread()
  else:
    th = current_thread
  ## Return requested information
  return '%s\t%s' % (th.GetThreadID(), th.GetSelectedFrame().GetFunctionName())


## FUNCTION get_current_frame_info:
## --------------------------------

def get_current_frame_info():
  """
  Returns the information about the current frame of execution: program counter, function
  name, file name and line number. Called by the 'getfrm' command received through the
  socket.
  
  @rtype: string
  """
  ## Get current thread
  if current_thread is None:
    current_frame = current_process.GetSelectedThread().GetSelectedFrame()
  else:
    current_frame = current_thread.GetSelectedFrame()
  ## Return requested information
  line_entry = current_frame.GetLineEntry()
  return '%s\t%s\t%s\t%s' % (current_frame.GetPC(), current_frame.GetFunctionName(), line_entry.GetFileSpec(), line_entry.GetLine())


## FUNCTION call_function:
## -----------------------

def call_function(function_name):
  """
  Calls a function in the debugged program. Called by the 'callfunc' command received
  through the socket.
  
  Actually does nothing here, since it never seems to be called.
  """
  return None


## FUNCTION get_local_variables:
## -----------------------------

def get_local_variables():
  """
  Returns the names for the local variables in the current execution frame. The names are
  in a space-separated string. Called by the 'locvars' command received through the socket.
  
  @rtype: string
  """
  ## Get current thread
  if current_thread is None:
    current_frame = current_process.GetSelectedThread().GetSelectedFrame()
  else:
    current_frame = current_thread.GetSelectedFrame()
  ## Get all variables and return their name in a space-separated list
  ## NB: the boolean parameters are:
  ##  - include locals, which we want;
  ##  - include parameters, which we want;
  ##  - include static variables, which we don't want;
  ##  - in local scope only, which we want, or we'll also see variables in blocks in macros.
  return ' '.join(v.name for v in current_frame.GetVariables(True, True, False, True))


## FUNCTION get_variable_value:
## ----------------------------

def get_variable_value(variable_name):
  """
  Returns the value for a variable as a string. Called by the 'gvarval' command received
  through the socket.
  
  @param variable_name: Name of the variable.
  @type variable_name: string
  
  @rtype: string
  """
  ## Get current thread
  if current_thread is None:
    current_frame = current_process.GetSelectedThread().GetSelectedFrame()
  else:
    current_frame = current_thread.GetSelectedFrame()
  ## GetValueForVariablePath only works for locals; for globals, we have to use CreateValueFromExpression
  var = current_frame.GetValueForVariablePath(variable_name)
  if var.value is None:
    var = current_target.CreateValueFromExpression(variable_name, variable_name)
  ## The variable value can contain any character, so we actually return the value encoded in hexa
  return S(codecs.encode(B(str(var.value)), 'hex_codec'))


## FUNCTION set_variable_value:
## ----------------------------

def set_variable_value(variable_name, value):
  """
  Changes the value of a variable in the current execution frame. Called by the 'svarval'
  command received through the socket.
  
  @param variable_name: Name of the variable to set.
  @type variable_name: string
  
  @param value: Hex-encoded new value for the variable.
  @type value: string
  """
  ## Get current thread
  if current_thread is None:
    current_frame = current_process.GetSelectedThread().GetSelectedFrame()
  else:
    current_frame = current_thread.GetSelectedFrame()
  ## Get variable
  ## NB: the case where the variable does not exist in the current scope is not considered,
  ## as the following won't trigger any error if the variable does not exist, and there
  ## doesn't seem to be any way to set the value for a global variable anyway.
  var = current_frame.GetValueForVariablePath(variable_name)
  ## Set variable value
  var.value = S(codecs.decode(value, 'hex_codec'))
  return None


## FUNCTION get_variable_type_info:
## --------------------------------

_variable_types = {
  lldb.eTypeClassArray              : 'array',
  lldb.eTypeClassBuiltin            : 'normal',
  lldb.eTypeClassPointer            : 'pointer',
  lldb.eTypeClassStruct             : 'struct',
  lldb.eTypeClassUnion              : 'union'
}
def get_variable_type_info(variable_name):
  """
  Returns the information on the type of a given variable: type kind, type name and number
  of elements for an array. Called by the 'vartyp' command received through the socket.
  
  @param variable_name: Name of the variable.
  @type variable_name: string
  
  @rtype: string
  """
  ## Get current thread
  if current_thread is None:
    current_frame = current_process.GetSelectedThread().GetSelectedFrame()
  else:
    current_frame = current_thread.GetSelectedFrame()
  ## Get variable (which has to be local)
  var = current_frame.GetValueForVariablePath(variable_name)
  ## Get variable type, resolving typedef's
  var_type = var.type
  while var_type.type == lldb.eTypeClassTypedef:
    var_type = var_type.GetTypedefedType()
  ## Get type kind for its type
  var_type_kind = _variable_types.get(var_type.type, 'unknown')
  ## If variable is a pointer, adapt its kind depending on what it points to
  if var_type_kind == 'pointer':
    pointed_var = current_frame.GetValueForVariablePath('*%s' % variable_name)
    pointed_var_type = pointed_var.type
    while pointed_var_type.type == lldb.eTypeClassTypedef:
      pointed_var_type = pointed_var_type.GetTypedefedType()
    pointed_var_type_kind = _variable_types.get(pointed_var_type.type, 'unknown')
    if pointed_var_type_kind == 'struct':
      var_type_kind = 'pstruct'
    elif pointed_var_type_kind == 'union':
      var_type_kind = 'punion'
    elif pointed_var_type_kind == 'pointer':
      var_type_kind = 'ppointer'
  ## Return requested information
  return '%s\t%s\t%s' % (var_type_kind, var.type.name, var.GetNumChildren() if var_type_kind == 'array' and var.MightHaveChildren() else -1)


## FUNCTION get_variable_fields:
## -----------------------------

def get_variable_fields(variable_name):
  """
  Returns the names of the fields in a given variable as a space-separated list. Called
  by the 'varflds' command received through the socket.
  
  @param variable_name: Name of the variable.
  @type variable_name: string
  
  @rtype: string
  """
  ## Get current thread
  if current_thread is None:
    current_frame = current_process.GetSelectedThread().GetSelectedFrame()
  else:
    current_frame = current_thread.GetSelectedFrame()
  ## Get variable type (variable has to be local)
  var_type = current_frame.GetValueForVariablePath(variable_name).type
  while True:
    if var_type.type == lldb.eTypeClassTypedef:
      var_type = var_type.GetTypedefedType()
    elif var_type.type == lldb.eTypeClassPointer:
      var_type = var_type.GetPointeeType()
    else:
      break
  ## Return 'members' of the type, i.e its fields
  return ' '.join(m.name for m in var_type.members)


## FUNCTION read_memory:
## ---------------------

def read_memory(length, address, format):
  """
  Reads a given number of bytes in the executed program's memory, starting at a given
  address. Called by the 'rdmem' command received through the socket.
  
  Actually does nothing here, since it never seems to be called.
  """
  return None


## FUNCTION reset_system:
## ----------------------

def reset_system():
  """
  Resets the running system, restarting the program execution from the beginning. Called
  by the 'reset' command received through the socket.
  """
  global current_target
  ## Kill and detach current process
  current_process.Kill()
  current_process.Detach()
  ## Delete current target
  lldb.debugger.DeleteTarget(current_target)
  ## Recreate target by reloading the debugged program
  current_target = lldb.debugger.CreateTarget(program_executable)
  ## Return answer indication reset has been done
  return "reset done"


## MAIN FUNCTION run:
## ==================

_command_mapping = {
  'ldpgm'     : load_program,
  'sbpfunc'   : set_breakpoint_on_function,
  'sbpfile'   : set_breakpoint_on_file,
  'delbp'     : delete_breakpoint,
  'run'       : run_program,
  'cont'      : continue_program,
  'step'      : step_over,
  'stepin'    : step_in,
  'stepout'   : step_out,
  'stop'      : stop_program,
  'getth'     : get_active_thread_info,
  'getfrm'    : get_current_frame_info,
  'callfunc'  : call_function,
  'locvars'   : get_local_variables,
  'gvarval'   : get_variable_value,
  'svarval'   : set_variable_value,
  'vartyp'    : get_variable_type_info,
  'varflds'   : get_variable_fields,
  'rdmem'     : read_memory,
  'reset'     : reset_system
}

def run():
  """
  Main socket server loop, getting commands on the socket and sending back the answers
  when needed.
  """
  global client_socket
  ## Create server socket
  s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
  s.bind(('', int(os.environ['RTDS_LLDB_PORT_NUMBER'])))
  s.listen(1)
  ## Wait for a connection
  client_socket, addr = s.accept()
  ## Command execution loop
  while True:
    ## Read socket until a line feed is received, indication the end of a command
    command_and_params = [B('')]
    while True:
      c = client_socket.recv(1)
      if c == B('\n'):
        break
      elif c == B('\t'):
        command_and_params.append(B(''))
      else:
        command_and_params[-1] += c
    command_and_params = [S(z) for z in command_and_params]
    if DEBUG:
      print('> %s' % ' '.join(command_and_params))
    ## If command is 'leave', treat it here
    if command_and_params[0] == 'leave':
      ## Terminate current process
      if current_process is not None:
        current_process.Kill()
        current_process.Detach()
      ## Reset debugger
      lldb.debugger.Clear()
      ## Get out of command execution loop
      break
    ## If command is a known one
    elif command_and_params[0] in _command_mapping:
      ## Execute it and get the info to send back to the client
      answer = _command_mapping[command_and_params[0]](*command_and_params[1:])
      ## If there is any info to send back, send it
      if answer is not None:
        client_socket_lock.acquire()
        client_socket.send(B(answer + '\n'))
        client_socket_lock.release()
  ## If we get there, it's a leave: close socket connection & quit
  client_socket.close()
  quit()
