#define __STDC_FORMAT_MACROS
#include <string.h>
#include <inttypes.h>

#include "RTDS_Scheduler.h"
#include "RTDS_Proc.h"
#include "RTDS_InstanceManager.h"
#include "RTDS_Deployment.h"
#include "RTDS_Common.h"
#include "RTDS_BasicTypes.h"
#include "RTDS_InternalConstants.h"
#include "RTDS_Proxy.h"
#include "RTDS_OS.h"
#include "RTDS_gen.h"


RTDS_Scheduler::RTDS_Scheduler(void *queueId)
  {
  component = NULL;
  nextInstanceNumber = 0;
  scheduledInstances = NULL;
  timerList = NULL;
  readInputQueue = NULL;
  writeInputQueue = NULL;
  envInstance = NULL;
  proxyInstance = new RTDS_Proxy(this);
  }

RTDS_Scheduler::~RTDS_Scheduler() 
  {
  // Clean instance list
  while (scheduledInstances != NULL)
    {
    RTDS_ScheduledInstance *instance = scheduledInstances;
    scheduledInstances = scheduledInstances->next;
    delete instance->instanceObject;
    free(instance);
    }
  // Clean timer list
  while (timerList != NULL)
    {
    RTDS_TimerState *timer = timerList;
    timerList = timerList->next;
    free(timer);
    }
  // Clean input queue
  while (readInputQueue != NULL)
    {
    RTDS_MessageHeader *message = readInputQueue;
    readInputQueue = readInputQueue->next;
    if (message->pData != NULL)
      {
      free(message->pData);
      }
    free(message);
    }
  // Delete proxy
  delete proxyInstance;
  }

void RTDS_Scheduler::sendMessage(RTDS_SdlInstanceId &sender, RTDS_SdlInstanceId &receiver, int messageNumber, long dataLength, unsigned char *pData, int timerUniqueId) 
  {
  // Create message
  RTDS_MessageHeader *message = (RTDS_MessageHeader *) malloc(sizeof(RTDS_MessageHeader));
  message->messageNumber = messageNumber;
  if (messageNumber > 0 && timerUniqueId == 0)
    {
    message->messageUniqueId = RTDS_GetMessageUniqueId();
    }
  else
    {
    message->messageUniqueId = 0;
    }
  message->timerUniqueId = timerUniqueId;
  message->sender = sender;
  message->receiver = receiver;
  message->dataLength = dataLength;
  message->pData = pData;
  message->next = NULL;
  // If receiver is not defined (i.e., send by name)
  if (message->receiver.instanceNumber == 0)
    {
    // Look in local scheduler
    RTDS_ScheduledInstance *instance;
    for (instance = scheduledInstances; instance != NULL; instance = instance->next)
      {
      RTDS_SdlInstanceId &sdlInstanceId = instance->instanceObject->RTDS_currentContext->mySdlInstanceId;
      if (sdlInstanceId.processNumber == message->receiver.processNumber)
        {
        message->receiver = sdlInstanceId;
        break;
        }
      }
    // If receiver wasn't found on local scheduler
    if (instance == NULL)
      {
      // Look in other schedulers (i.e, components)
      RTDS_DeplComponent *comp = component->node->getComponent(message->receiver.processNumber, component);
      // Scheduler found
      if (comp != NULL)
        {
        message->receiver.componentNumber = comp->componentUniqueId;
        }
      // Scheduler was not found
      else
        {
        message->receiver.componentNumber = 0;
        }
      }
    }
  // Enqueue message
  if (writeInputQueue == NULL)
    {
    writeInputQueue = message;
    readInputQueue = message;
    }
  else
    {
    writeInputQueue->next = message;
    writeInputQueue = message;
    }
  // Trace
  if (messageNumber > 0 && timerUniqueId == 0) 
    {
    if (sender.componentNumber != component->componentUniqueId)
      {
      sender = proxyInstance->RTDS_currentContext->mySdlInstanceId;
      }
    fprintf(RTDS_traceFile, "    <messageSent nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" pName=\"p%d\" mId=\"%d\" pId=\"%u\" sigNum=\"%d\" msgName=\"m%d\" />\n",
      component->node->nodeUniqueId, 
      component->componentUniqueId,
      RTDS_TickGet(), 
      sender.processNumber, 
      message->messageUniqueId, 
      sender.instanceNumber, 
      messageNumber, 
      messageNumber
    );
    }
  }
	
RTDS_SdlInstanceId &RTDS_Scheduler::createInstance(int processNumber, RTDS_Proc *parent, RTDS_Proc *self) 
  {
  // Create instance if necessary
  if (self == NULL)
    {
    self = RTDS_InstanceManager::createInstance(this, processNumber);
    if (processNumber == RTDS_process_RTDS_Env)
      {
      envInstance = self;
      }
    }
  // Setup context
  RTDS_GlobalProcessInfo *context = (RTDS_GlobalProcessInfo *) malloc(sizeof(RTDS_GlobalProcessInfo));
  context->mySdlInstanceId.processNumber = processNumber;
  context->mySdlInstanceId.instanceNumber = ++nextInstanceNumber;
  context->mySdlInstanceId.componentNumber = component->componentUniqueId;
  if (parent != NULL)
    {
    context->parentSdlInstanceId = parent->RTDS_currentContext->mySdlInstanceId;
    }
  context->currentMessage = NULL;
  context->readSaveQueue = NULL;
  context->writeSaveQueue = NULL;
  context->sdlState = 0;
  self->RTDS_currentContext = context;
  // Create scheduled instance
  RTDS_ScheduledInstance *newInstance = (RTDS_ScheduledInstance *) malloc(sizeof(RTDS_ScheduledInstance));
  newInstance->instanceObject = self;
  newInstance->next = NULL;
  if (scheduledInstances == NULL)
    {
    scheduledInstances = newInstance;
    }
  else
    {
    RTDS_ScheduledInstance *instance;
    for (instance = scheduledInstances; instance->next != NULL; instance = instance->next);
    instance->next = newInstance;
    }
  // Trace
  if (context->mySdlInstanceId.processNumber < 0)
    {
    fprintf(RTDS_traceFile, "    <semaphoreCreated nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" semName=\"x%d\" stillAvailable=\"%d\" pId=\"%u\" />\n",
      component->node->nodeUniqueId,
      component->componentUniqueId,
      RTDS_TickGet(), 
      -context->mySdlInstanceId.processNumber, 
      ((RTDS_SemaphoreProcess *) self)->isAvailable(), 
      context->mySdlInstanceId.instanceNumber
    );
    } 
  else 
    {
    RTDS_SdlInstanceId &creator = parent != NULL ? context->parentSdlInstanceId : context->mySdlInstanceId;
    fprintf(RTDS_traceFile, "    <taskCreated nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" creatorId=\"%u\" pName=\"p%d\" creatorName=\"p%d\" pId=\"%u\" />\n",
      component->node->nodeUniqueId,
      component->componentUniqueId,
      RTDS_TickGet(), 
      creator.instanceNumber, 
      context->mySdlInstanceId.processNumber, 
      creator.processNumber, 
      context->mySdlInstanceId.instanceNumber
    );
    }
  // Send startup message
  sendMessage(context->mySdlInstanceId, context->mySdlInstanceId, RTDS_message_RTDS_startProcess, 0, NULL);
  // Return sdl instance
  return context->mySdlInstanceId;
  }
  
void RTDS_Scheduler::run() 
  {
  // Start proxy
  proxyInstance->RTDS_executeTransition(NULL);
  // Loop forever
  while (1)
    {
    while (readInputQueue != NULL)
      {
      // Pop first message from queue
      RTDS_MessageHeader *message = readInputQueue;
      readInputQueue = readInputQueue->next;
      if (readInputQueue == NULL)
        {
        writeInputQueue = NULL;
        }
      // Find receiver
      RTDS_ScheduledInstance *instance = NULL;
      RTDS_Proc *receiver = NULL;
      // Local message
      if (message->receiver.componentNumber == component->componentUniqueId)
        {
        // Look for receiver in local scheduler
        for (instance = scheduledInstances; instance != NULL; instance = instance->next)
          {
          if (instance->instanceObject->RTDS_currentContext->mySdlInstanceId.instanceNumber == message->receiver.instanceNumber)
            {
            receiver = instance->instanceObject;
            break;
            }
          }
        // No receiver, discard message
        if (receiver == NULL)
          {
          if (message->pData != NULL)
            {
            free(message->pData);
            }
          free(message);
          ns3::Simulator::ScheduleNow(&RTDS_Scheduler::run, this);
          return;
          }
        }
      // Distributed message
      else
        {
        // Scheduler is available
        if (message->receiver.componentNumber > 0)
          {
          // Receiver is the local proxy
          receiver = proxyInstance;
          }
        // Scheduler is not available
        else
          {
          // No receiver, discard message
          if (message->pData != NULL)
            {
            free(message->pData);
            }
          free(message);
          ns3::Simulator::ScheduleNow(&RTDS_Scheduler::run, this);
          return;
          }
        }
      // Startup message
      if (message->messageNumber == RTDS_message_RTDS_startProcess)
        {
        free(message);
        message = NULL;
        }
      // Context
      RTDS_GlobalProcessInfo *context = receiver->RTDS_currentContext;
      // Trace
      if (message != NULL)
        {
        if (message->messageNumber == RTDS_message_RTDS_takeSemaphore) 
          {
          for (instance = scheduledInstances; instance != NULL; instance = instance->next)
            {
            if (instance->instanceObject->RTDS_currentContext->mySdlInstanceId.instanceNumber == message->sender.instanceNumber)
              {
              fprintf(RTDS_traceFile, "    <takeAttempt nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" pName=\"p%d\" semName=\"x%d\" timeout=\"%d\" pId=\"%u\" semId=\"%u\" />\n",
                component->node->nodeUniqueId, 
                component->componentUniqueId,
                RTDS_TickGet(), 
                message->sender.processNumber, 
                -context->mySdlInstanceId.processNumber, 
                ((RTDS_SemaphoreTakeProcedure *) instance->instanceObject->RTDS_calledProcedure)->timeOut, 
                message->sender.instanceNumber, 
                context->mySdlInstanceId.instanceNumber
              );
              break;
              }
            }
          }
        else if (message->messageNumber > 0) 
          {
          if (message->timerUniqueId != 0) 
            {
            fprintf(RTDS_traceFile, "    <timerTimedOut nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" pName=\"p%d\" timerName=\"m%d\" pId=\"%u\" tId=\"%d\" />\n",
              component->node->nodeUniqueId, 
              component->componentUniqueId,
              RTDS_TickGet(), 
              context->mySdlInstanceId.processNumber, 
              message->messageNumber, 
              context->mySdlInstanceId.instanceNumber, 
              message->timerUniqueId
            );
            }
          else 
            {
            fprintf(RTDS_traceFile, "    <messageReceived nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" pName=\"p%d\" mId=\"%d\" pId=\"%u\" sigNum=\"%d\" msgName=\"m%d\" />\n",
              component->node->nodeUniqueId, 
              component->componentUniqueId,
              RTDS_TickGet(),
              context->mySdlInstanceId.processNumber, 
              message->messageUniqueId, 
              context->mySdlInstanceId.instanceNumber, 
              message->messageNumber, 
              message->messageNumber
            ); 
            }
          }
        }
      // Remember state and execute transition
      int initialState = context->sdlState;
      short killed = receiver->RTDS_executeTransition(message);
      // If process was not killed, execute transitions for continuous signals in current state
      if (!killed)
        {
        int lowestPriority;
        int previousState = -1;
        while (!killed && context->sdlState != previousState)
          {
          previousState = context->sdlState;
          // Execute all transition for continuous signals, starting with the one with the highest priority,
          // until there are no more transitions to execute or instance is killed
          lowestPriority = 0;
          for (;;)
            {
            int previousLowestPriority = lowestPriority;
            // If we're in SDL semantics, if there is any message for current instance in the scheduler's queue,
            // stop considering continuous signals
#ifdef RTDS_SDL_SEMANTICS
            for (message = readInputQueue; message != NULL; message = message->next)
              {
              if (message->receiver.componentNumber == component->componentNumber && message->receiver.instanceNumber == context->mySdlInstanceId.instanceNumber)
                {
                break;
                }
              }
            if (message != NULL)
              {
              break;
              }
#endif
            killed = receiver->RTDS_continuousSignals(&lowestPriority);
            // Done if instance changed its state, was killed, or if no more continuous signal
            if (context->sdlState != previousState || killed || lowestPriority == previousLowestPriority)
              {
              break;
              }
            }
          }
        }
      // Process killed
      if (killed)
        {
        // Remove timers
        RTDS_TimerState *currTimer = timerList, *prevTimer = NULL;
        while (currTimer != NULL)
          {
          if (currTimer->receiverId.instanceNumber == context->mySdlInstanceId.instanceNumber)
            {
            if (prevTimer == NULL)
              {
              timerList = currTimer->next;
              free(currTimer);
              currTimer = timerList;
              }
            else
              {
              prevTimer->next = currTimer->next;
              free(currTimer);
              currTimer = prevTimer->next;
              }
            }
          else
            {
            currTimer = currTimer->next;
            }
          }
        // Remove scheduled instance
        RTDS_ScheduledInstance *currInstance, *prevInstance = NULL;
        for (currInstance = scheduledInstances; currInstance != NULL; prevInstance = currInstance, currInstance = currInstance->next)
          {
          if (currInstance == instance)
            {
            if (prevInstance == NULL)
              {
              scheduledInstances = currInstance->next;
              }
            else
              {
              prevInstance->next = currInstance->next;
              }
            // Trace
            fprintf(RTDS_traceFile, "    <taskDeleted nId=\"n%u\" cId=\"c%u\" time=\"%"PRId64"\" pName=\"p%d\" pId=\"%u\" />\n",
              component->node->nodeUniqueId,
              component->componentUniqueId,
              RTDS_TickGet(),
              currInstance->instanceObject->RTDS_currentContext->mySdlInstanceId.processNumber,
              currInstance->instanceObject->RTDS_currentContext->mySdlInstanceId.instanceNumber
            );
            if (currInstance->instanceObject == envInstance)
              {
              envInstance = NULL;
              }
            delete currInstance->instanceObject;
            free(currInstance);
            break;
            }
          }
        }
      // Process alive: handle save queue
      else
        {
        if (context->sdlState != initialState && context->readSaveQueue != NULL)
          {
          // Copy save queue to input queue
          context->writeSaveQueue->next = readInputQueue;
          readInputQueue = context->readSaveQueue;
          // Empty save queue
          context->readSaveQueue = NULL;
          context->writeSaveQueue = NULL;
          }
        }
      // Loop
      ns3::Simulator::ScheduleNow(&RTDS_Scheduler::run, this);
      return;
      }
    // Handle timers
    if (timerList != NULL)
      {
      RTDS_TimerState *timer = timerList;
      long long timeLeft = timer->timeoutValue - RTDS_TickGet();
      // Not expired
      if (timeLeft > 0)
        {
        ns3::Simulator::Schedule(ns3::NanoSeconds(timeLeft), &RTDS_Scheduler::run, this);
        }
      // Expired
      else
        {
        timerList = timerList->next;
        sendMessage(timer->receiverId, timer->receiverId, timer->timerNumber, 0, NULL, timer->timerNumber);
        free(timer);
        ns3::Simulator::ScheduleNow(&RTDS_Scheduler::run, this);
        }
      }
    // Neither messages nor timers left
    else
      {
      // Look for any running instances except environment (in all nodes and components)
      RTDS_DeplNodeList *nodeElement;
      RTDS_DeplComponentList *componentElement;
      RTDS_ScheduledInstance *instance;
      for (nodeElement = RTDS_DeplNode::nodeList; nodeElement != NULL; nodeElement = nodeElement->next)
        {
        for (componentElement = nodeElement->node->componentList; componentElement != NULL; componentElement = componentElement->next)
          {
          for (instance = componentElement->component->scheduler->scheduledInstances; instance != NULL; instance = instance->next)
            {
            // Check for instance (skip environment and semaphores)
            if (instance->instanceObject != componentElement->component->scheduler->envInstance 
              && instance->instanceObject->RTDS_currentContext->mySdlInstanceId.processNumber > 0)
              {
              return;
              }
            }
          }
        }
      // No instances left, terminate simulation
      ns3::Simulator::Stop();
      }
    // Always exit the loop
    break;
    }
  }
