#ifndef _TCI_H
#define _TCI_H

#include <RTDS_BasicTypes.h>
#include <RTDS_Common.h>

#ifdef __cplusplus
extern "C" {
#endif

#define TCI_VERDICT_NONE 0
#define TCI_VERDICT_PASS 1
#define TCI_VERDICT_INCONC 2
#define TCI_VERDICT_FAIL 3
#define TCI_VERDICT_ERROR 4

#define TRI_ERROR -1
#define TRI_OK 0
#define TCI_ERROR -1
#define TCI_OK 0

typedef unsigned char Boolean;

typedef struct _QualifiedName
{
//moduleName: module name of the TTCN-3 module.
//baseName: name of the object that is fully qualified.
//aux: for future extensibility of TRI.
char* moduleName;
char* baseName;
void* aux;
}QualifiedName;

typedef struct _BinaryString
{
//data: the string.
//bits: number of bits used for data (-1 for omited value).
//aux: for future extensibility of TRI.
unsigned char* data;
long int bits;
void* aux;
} BinaryString;

/* temp type*/
//from tri

typedef BinaryString TriAddress;
typedef BinaryString TriMessage;
typedef BinaryString TriException;
typedef BinaryString TriTimerId;

typedef QualifiedName TriFunctionId;
typedef QualifiedName TriSignatureId;
typedef QualifiedName TriTestCaseId;

typedef unsigned long TriTimerDuration;
typedef long int TriStatus;

typedef struct _TriAddressList
{
TriAddress** addrList;
long int length;
}TriAddressList;

typedef struct _TriComponentId
{
BinaryString compInst;
char* compName;
QualifiedName compType;
}TriComponentId;

typedef struct _TriComponentIdList
{
TriComponentId** addrList;
long int length;
}TriComponentIdList;

typedef enum
{
TRI_IN_PAR = 0,
TRI_INOUT_PAR = 1,
TRI_OUT_PAR = 2
} TriParameterPassingMode;

typedef struct _TriParameter
{
BinaryString par;
TriParameterPassingMode mode;
}TriParameter;

typedef struct _TriParameterList
{
TriParameter** parameterList;
long int length;
}TriParameterList;

typedef struct _TriPortId
{
TriComponentId * compInst;
char* portName;
long int portIndex;
QualifiedName portType;
void* aux;
}TriPortId;

typedef struct _TriPortIdList
{
TriPortId** portIdList;
long int length;
}TriPortIdList;








/****************************************************************************************************************************/

//abstract type
typedef void* TciValue;
typedef void* tciValue;
typedef void* TciType;
typedef void* Type;
typedef void* TciValueTemplate;
typedef void* Value;
typedef void* TciStatus;
typedef void* TciNonValueTemplate;

/****************************************************************************************************************************/

typedef int VerdictValue;
typedef int verdicttype;

typedef QualifiedName TciBehaviourIdType;
typedef QualifiedName TciModuleIdType;
typedef QualifiedName TciModuleParameterIdType;
typedef QualifiedName TciTestCaseIdType;

typedef struct _TciModuleIdListType
{
long int length;
TciModuleIdType *modList;
} TciModuleIdListType;

typedef struct _TciTestCaseIdListType
{
long int length;
TciTestCaseIdType *testcaseList;
} TciTestCaseIdListType;

typedef struct _TciModuleParameterType
{
char* parName;
TciValue defaultValue;
} TciModuleParameterType;

typedef struct _TciModuleParameterListType
{
long int length;
TciModuleParameterType *modParList;
} TciModuleParameterListType;

typedef enum
{
TCI_IN_PAR = 0,
TCI_INOUT_PAR = 1,
TCI_OUT_PAR = 2
} TciParameterPassingModeType;

typedef struct _TciParameterType
{
char* parName;
TciParameterPassingModeType parPassMode;
TciValue parValue;
} TciParameterType;

typedef struct _TciParameterListType
{
long int length;
TciParameterType *parList;
} TciParameterListType;

typedef struct _TciParameterTypeType
{
Type parType;
TciParameterPassingModeType parPassMode;
} TciParameterTypeType;

typedef struct _TciParameterTypeListType
{
long int length;
TciParameterTypeType *typeList;
} TciParameterTypeListType;

typedef enum
{
TCI_CTRL_COMP,
TCI_MTC_COMP,
TCI_PTC_COMP,
TCI_SYS_COMP,
TCI_ALIVE_COMP
} TciTestComponentKindType;

typedef enum
{
TCI_ADDRESS_TYPE,
TCI_ANYTYPE_TYPE,
TCI_BITSTRING_TYPE,
TCI_BOOLEAN_TYPE,
TCI_CHAR_TYPE,
TCI_CHARSTRING_TYPE,
TCI_COMPONENT_TYPE,
TCI_ENUMERATED_TYPE,
TCI_FLOAT_TYPE,
TCI_HEXSTRING_TYPE,
TCI_INTEGER_TYPE,
TCI_OBJID_TYPE,
TCI_OCTETSTRING_TYPE,
TCI_RECORD_TYPE,
TCI_RECORD_OF_TYPE,
TCI_SET_TYPE,
TCI_SET_OF_TYPE,
TCI_UNION_TYPE,
TCI_UNIVERSAL_CHAR_TYPE,
TCI_UNIVERSAL_CHARSTRING_TYPE,
TCI_VERDICT_TYPE
} TciTypeClassType;

typedef enum
{
inactiveC,
runningC,
stoppedC,
killedC
} ComponentStatusType;

typedef enum
{
runningT,
inactiveT,
expiredT
} TimerStatusType;

typedef enum
{
startedP,
haltedP,
stoppedP
} PortStatusType;

typedef struct _TciValueDifference
{
TciValue val;
TciValueTemplate tmpl;
char* desc;
} TciValueDifference;

typedef struct _TciValueDifferenceList
{
long int length;
TciValueDifference* diffList;
} TciValueDifferenceList;

/*********************************************************************************************************************************************/
/*
 *TCI-TM required operation
 */
/*********************************************************************************************************************************************/

void tciRootModule(TciModuleIdType* moduleId);
  /*
  Select root module.
  
  @param moduleID: module identifier.
  */

TciModuleIdListType* tciGetImportedModules();
  /*
  Return the list of all imported module in previously selected module.
  
  @return: list of all imported moduless, empty module list if no imported module.
  */

TciModuleParameterListType* tciGetModuleParameters(TciModuleIdType* moduleName);
  /*
  Return the list of parameters of the identified module.
  
  @param moduleName: 
  
  @return: list of all module parameters of identified module.
  */

TciTestCaseIdListType* tciGetTestCases();
  /*
  Return the list of test cases in the selected root module.
  
  @return: list of all test cases defined in and imported into root module.
  */

TciParameterTypeListType* tciGetTestCaseParameters(TciTestCaseIdType* testCaseId);
  /*
  Return the list of parameters of the identified test case.
  
  @param testCaseId: test case identifier.
  
  @return: list of all module parameters of identified test case, empty list if no parameter.
  */

TriPortIdList* tciGetTestCaseTSI(TciTestCaseIdType* testCaseId);
  /*
  Return the list of ports of the identified test case.
  
  @param testCaseId: test case identifier.
  
  @return: list of all system ports of the TSI (of the MTC if no TSI has been declared), empty list if no system port.
  */

void tciStartTestCase(TciTestCaseIdType* testCaseId, TciParameterListType* parameterlist);
  /*
  Start the test case identified.
  
  @param testCaseId: test case identifier.
  
  @param parameterlist: list of values, each value defines a parameter. The parameters are ordered as described in TTCN-3 declaration. Null or an empty list can be passed if no parameter.
  */

void tciStopTestCase();
  /*
  Stop the previously started test case.
  */

TriComponentId* tciStartControl();
  /*
  Start the control part of the selected module and return the id of the control component.
  
  @return: the test component the module control part is running on. null if TE can not start control part.
  */

void tciStopControl();
  /*
  Stop execution of the control part.
  */
  
/*********************************************************************************************************************************************/
/*
 *TCI-TM provided operation
 */
/*********************************************************************************************************************************************/
 
void tciTestCaseStarted(TciTestCaseIdType* testCaseId, TciParameterListType* parameterList, double timer);
  /*
  Indicates to the TM the test case testCaseId has been started.
  
  @param testCaseId: test case identifier.
  
  @param parameterList: list of values ordered as described in TTCN-3 test case declaration.
  
  @param timer: dureation of test case timer.
  */

void tciTestCaseTerminated(VerdictValue verdict, TciParameterListType* parameterlist);
  /*
  Indicates to the TM the previously started test case has been executed.
  
  @param verdict: final verdict of test case
  
  @param parameterlist: list of values ordered as described in TTCN-3 test case declaration.
  */

void tciControlTerminated();
  /*
  Indicates to the TM the control part has been executed.
  */

tciValue* tciGetModulePar(TciModuleParameterIdType* parameterId);
  /*
  Return the value of the indicated parameter.
  
  @param parameterId: module parameter identifier.
  
  @return: the value of the parameter.
  */
  
void tciLog(char* message);
  /*

  */

void tciError(char* message);
  /*
  Indicate to the TM an unrecoverable error.
  
  @param message: the message error.
  */
/*********************************************************************************************************************************************/
/*
 *TCI-CD required operation
 */
/*********************************************************************************************************************************************/
 
TciType tciGetTypeForName(char* typeName);
  /*
  Return a type representing TTCN-3 type.
  
  @param typeName: TTCN-3 type name.
  
  @return: type representing TTCN-3 type.
  */ 

TciType* tciGetInteger();
  /*
  Constructs and returns TTCN-3 integer type.
  
  @return: an instance representing a TTCN-3 integer type.
  */

TciType* tciGetFloat();
  /*
  Constructs and returns TTCN-3 float type.
  
  @return: an instance representing a TTCN-3 float type.
  */

TciType* tciGetBoolean();
  /*
  Constructs and returns TTCN-3 boolean type.
  
  @return: an instance representing a TTCN-3 boolean type.
  */

TciType* tciGetChar();
  /*
  Constructs and returns TTCN-3 char type.
  
  @return: an instance representing a TTCN-3 char type.
  */

TciType* tciGetUniversalChar();
  /*
  Constructs and returns TTCN-3 universal char type.
  
  @return: an instance representing a TTCN-3 universal char type.
  */

TciType* tciGetTciObjid();
  /*
  Constructs and returns TTCN-3 object id type.
  
  @return: an instance representing a TTCN-3 object id type.
  */

TciType* tciGetTciCharstring();
  /*
  Constructs and returns TTCN-3 charstring type.
  
  @return: an instance representing a TTCN-3 charstring type.
  */

TciType* tciGetUniversalCharstring();
  /*
  Constructs and returns TTCN-3 universal charstring type.
  
  @return: an instance representing a TTCN-3 universal charstring type.
  */

TciType* tciGetHexstring();
  /*
  Constructs and returns TTCN-3 hexstring type.
  
  @return: an instance representing a TTCN-3 hexstring type.
  */

TciType* tciGetBitstring();
  /*
  Constructs and returns TTCN-3 bitstring type.
  
  @return: an instance representing a TTCN-3 bitstring type.
  */

TciType* tciGetOctetstring();
  /*
  Constructs and returns TTCN-3 octetstring type.
  
  @return: an instance representing a TTCN-3 octetstring type.
  */

int tciGetVerdict();
  /*
  Constructs and returns TTCN-3 verdict type.
  
  @return: an instance representing a TTCN-3 verdict type.
  */

void tciErrorReq(char* message);
  /*
  Indicate to the TM an unrecoverable error within CD.
  
  @param message: error phrase.
  */

/*********************************************************************************************************************************************/
/*
 *TCI-CD provided operation
 */
/*********************************************************************************************************************************************/

TciValue* tciDecode(BinaryString* message, TciType* decHypothesis);
  /*
  Decode message and return TTCN-3 value.
  
  @param message: the message to be decoded.
  
  @param decHypothesis: the hypothesys the decoding can be based on.
  
  @return: decoded value, null if  value is not of a compatible type as the decHypothesis
  */
  
BinaryString* tciEncode(TciValue* value);
  /*
  Return encoded TriMessage.
  
  @param value:  the value to encode.
  
  @return: encoding message for the specified encoding rules.
  */
  
/*********************************************************************************************************************************************/
/*
 *TCI-CH required operation
 */
/*********************************************************************************************************************************************/
 
void tciEnqueueMsgConnected(TriPortId* sender, TriComponentId* receiver, TciValue* rcvdMessage);
  /*
  Enqueues the received value.
  
  @param sender: port identifier at the sending component.
  
  @param receiver: identifier of the receiving component.
  
  @param rcvdMessage: value to be enqueued.
  */

void tciEnqueueCallConnected(TriPortId* sender, TriComponentId* receiver, TriSignatureId* signature, TciParameterListType* parameterList);
  /*
  Call by the CH when a tciCallConnected has been called.
  
  @param sender: port identifier at the sending component.
  
  @param receiver: identifier of the receiving component.
  
  @param signature: identifier of the signature of the procedure call.
  
  @param parameterList: list of values which are part of the indicated signature.
  */

void tciEnqueueReplyConnected(TriPortId* sender, TriComponentId* receiver, TriSignatureId* signature, TciParameterListType parameterList, TciValue* returnValue);
  /*
  Call by the CH when a tciReplyConnected has been called
  
  @param sender: identifier of the port sending the reply.
  
  @param receiver: identifier of the receiving component.
  
  @param signature: identifier of the signature of the procedure call.
   
  @param parameterList: list of values which are part of the indicated signature.
  
  @param returnValue: (optional) return value.
  */

void tciEnqueueRaiseConnected(TriPortId* sender, TriComponentId* receiver, TriSignatureId* signature, TciValue* exception);
  /*
  Enqueues the exeception
  
  @param sender: identifier of the port sending the reply.
  
  @param receiver: identifier of the receiving component.
  
  @param signature: identifier of the signature of the procedure call.
   
  @param exception: the exception.
  */

TriComponentId* tciCreateTestComponent(TciTestComponentKindType* kind, TciType* componentType, char* name);
  /*
  TE creates a TTCN-3 component.
  
  @param kind: kinf of component (CONTROL, PTC or MTC).
  
  @param componentType: identifier of the component type that shall be created.
  
  @param name: name of the component.
  
  @return: a TriComponentId value for the created component.
  */

void tciStartTestComponent(TriComponentId* component, TciBehaviourIdType* behaviour, TciParameterListType* parameterList);
  /*
  TE starts behaviour of indicated component.
  
  @param component: identifier of the component to start (previously created by tciCreateTestComponent).
  
  @param behaviour: identifier of the behaviour to be started on test component.
  
  @param parameterList: list of values, each value defines a parameter. The parameters are ordered as described in TTCN-3 declaration. Null or an empty list can be passed if no parameter.
  */

void tciStopTestComponent(TriComponentId* component);
  /*
  TE stops behaviour of indicated component.
  
  @param component: identifier of the component to stop.
  */
  
void tciConnect(TriPortId* fromPort, TriPortId* toPort);
  /*
  TE connects the indicated ports.
  
  @param fromPort: identifer of the test component port to be connected from.
  
  @param toPort: identifer of the test component port to be connected to.
  */
  
void tciDisconnect(TriPortId* fromPort, TriPortId* toPort);
  /*
  TE disconnects the indicated ports.
  
  @param fromPort: identifer of the test component port to be disconnected.
  
  @param toPort: identifer of the test component port to be disconnected.
  */

void tciMap(TriPortId* fromPort, TriPortId* toPort);
  /*
  TE maps the indicated ports.
  
  @param fromPort: identifer of the test component port to be mapped from.
  
  @param toPort: identifer of the test component port to be mapped to.
  */


void tciUnmap(TriPortId* fromPort, TriPortId* toPort);
  /*
  TE unmaps the indicated ports.
  
  @param fromPort: identifer of the test component port to be unmapped.
  
  @param toPort: identifer of the test component port to be unmapped.
  */

void tciTestComponentTerminated(TriComponentId* component, VerdictValue verdict);
  /*
  TE is notified of the termination of a test component.
  
  @param component: identifier of the terminated component.
  
  @param verdict: verdict after termination.
  */
  
Boolean tciTestComponentRunning(TriComponentId* component);
  /*
  Return true if indicated component is running.
  
  @param component: identifier of the component to test.
  
  @return: true if the component is still running, false otherwise.
  */

Boolean tciTestComponentDone(TriComponentId* component);
  /*
  Return true if indicated component has completing execution.
  
  @param component: identifier of the component to test.
  
  @return: true if the component has complete execution, false otherwise.
  */

TriComponentId* tciGetMTC();
  /*
  Return component ID of the MTC if running on the local TE
  
  @return: a TriComponentId of the MTC if executing, null otherwise.
  */ 

void tciExecuteTestCase(TciTestCaseIdType* testCaseId, TriPortIdList* tsiPortList);
  /*
  TE determines static connections to the SUT.
  
  @param testCaseId: test case identifier.

  @param tsiPortList: list of all TSI ports if TSi has been definied, MTC ports otherwise. If no port, null value or empty list shall be passed.
  */
  
void tciReset();
  /*
  Reset test system
  */

void tciKillTestComponent(TriComponentId* component);
  /*
  TE stops behaviour on indicated component and transfers it into killed state.
  
  @param component: identifier of the component to kill.
  */

Boolean tciTestComponentAlive(TriComponentId* component);
  /*
  Return true if indicated component is alive.
  
  @param component: identifier of the component to test.
  
  @return: true if component is alive, false otherwise.
  */

Boolean tciTestComponentKilled(TriComponentId* component);
  /*
  Return true if indicated component is in killed state
  
  @param component: identifier of the component to test.
  
  @return: true if component has been killed, false otherwise.
  */

/*********************************************************************************************************************************************/
/*
 *TCI-CH provided operation
 */
/*********************************************************************************************************************************************/

void tciSendConnected(TriPortId* sender, TriComponentId* receiver, TciValue* sendMessage);
  /*
  Send asynchronous transmission to the given receiver component.
  
  @param sender: port identifier at the sending component.
  
  @param receiver: identifier of the receiving component.
  
  @param sendMessage: message to be sent.
  */

void tciSendConnectedBC(TriPortId* sender, TciValue* sendMessage);
  /*
  Send asynchronous transmission to all components connected to this port.
  
  @param sender: port identifier at the sending component.
  
  @param sendMessage: message to be sent.
  */

void tciSendConnectedMC(TriPortId* sender, TriComponentIdList* receivers, TciValue* sendMessage);
  /*
  Send asynchronous transmission to all given receiver components.
  
  @param sender: port identifier at the sending component.
  
  @param receivers: list of identifiers of receiving components.
  
  @param sendMessage: message to be sent.
  */

void tciCallConnected(TriPortId* sender, TriComponentId* receiver, TriSignatureId* signature, TciParameterListType* parameterList);
  /*
  Initiate procedure call, return whithout waiting for the return of the issued procedure call. CH transmits call to the TE on which receiver is executing.
  
  @param sender: port identifier at the sending component.
  
  @param receiver: identifier of the receiving component.
  
  @param signature: identifier of the signature of procedure call.
  
  @param parameterList: list of values which are part of the indicated signature.
  */

void tciCallConnectedBC(TriPortId* sender, TriSignatureId* signature, TciParameterListType* parameterList);
  /*
  Initiate procedure call, return whithout waiting for the return of the issued procedure call. CH transmits call to all TEs on which a receiver is executing.
  
  @param sender: port identifier at the sending component.
  
  @param signature: identifier of the signature of procedure call.
  
  @param parameterList: list of values which are part of the indicated signature.
  */

void tciCallConnectedMC(TriPortId* sender, TriComponentIdList* receivers, TriSignatureId* signature, TciParameterListType* parameterList);
  /*
  Initiate procedure call, return whithout waiting for the return of the issued procedure call. CH transmits call to all TEs on which a receiver is executing.
  
  @param sender: port identifier at the sending component.
  
  @param receivers: list of identifiers of receiving components.
  
  @param signature: identifier of the signature of procedure call.
  
  @param parameterList: list of values which are part of the indicated signature.
  */

void tciReplyConnected(TriPortId* sender, TriComponentId* receiver, TriSignatureId* signature, TciParameterListType* parameterList, TciValue* returnValue);
  /*
  CH issus reply to a procedure call. CH transmits the reply to the TE on which receiver is being executed.
  
  @param sender: identifier of the port sending the reply.
  
  @param receiver: identifier of component receiving the reply.
  
  @param signature: identifier of the signature of procedure call.
  
  @param parameterList: list of encoded parameters which are part of the indicated signature.
  
  @param returnValue: (optional) return value.
  */

void tciReplyConnectedBC(TriPortId* sender, TriSignatureId* signature, TciParameterListType* parameterList, TciValue* returnValue);
  /*
  CH issus reply to a procedure call. CH transmits the reply to all TEs on which receivers are being executed.
  
  @param sender: identifier of the port sending the reply.
  
  @param signature: identifier of the signature of procedure call.
  
  @param parameterList: list of encoded parameters which are part of the indicated signature.
  
  @param returnValue: (optional) return value.
  */

void tciReplyConnectedMC(TriPortId* sender, TriComponentIdList* receivers, TriSignatureId* signature, TciParameterListType* parameterList, TciValue* returnValue);
  /*
  CH issus reply to a procedure call. CH transmits the reply to all TEs on which receivers are being executed.
  
  @param sender: identifier of the port sending the reply.
  
  @param receivers: list of identifiers of components receiving reply.
  
  @param signature: identifier of the signature of procedure call.
  
  @param parameterList: list of encoded parameters which are part of the indicated signature.
  
  @param returnValue: (optional) return value.
  */

void tciRaiseConnected(TriPortId* sender, TriComponentId* receiver, TriSignatureId* signature, TciValue* exception);
  /*
  CH raise an exception to a procedure call. CH transmits the exception to the TE on which receiver is being executed.
  
  @param sender: identifier of the port sending the reply.
  
  @param receiver: identifier of component receiving the reply.
  
  @param signature: identifier of the signature of procedure call.
  
  @param exception: the exception.
  */

void tciRaiseConnectedBC(TriPortId* sender, TriSignatureId* signature, TciValue* exception);
  /*
  CH raise an exception to a procedure call. CH transmits the exception to all TEs on which receiver are being executed.
  
  @param sender: identifier of the port sending the reply.
  
  @param signature: identifier of the signature of procedure call.
  
  @param exception: the exception.
  */

void tciRaiseConnectedMC(TriPortId* sender, TriComponentIdList* receivers, TriSignatureId* signature, TciValue* exception);
  /*
  CH raise an exception to a procedure call. CH transmits the exception to all TEs on which receiver are being executed.
  
  @param sender: identifier of the port sending the reply.
  
  @param receivers: list of identifiers of components receiving reply.
  
  @param signature: identifier of the signature of procedure call.
  
  @param exception: the exception.
  */

TriComponentId tciCreateTestComponentReq(TciTestComponentKindType* kind, TciType* componentType, char* name);
  /*
  CH transmits component creation request to the remote TE, and calls tciCreateTestComponent to obtain component identifier.
  
  @param kind: kinf of component (CONTROL, PTC or MTC).
  
  @param componentType: identifier of the component type that shall be created.
  
  @param name: name of the component.
  
  @return: a TriComponentId value for the created component.
  */

void tciStartTestComponentReq(TriComponentId* component, TciBehaviourIdType* behaviour, TciParameterListType* parameterList);
  /*
  CH transmits start component request to the remote TE, and calls tciStartTestComponent.
  
  @param component: identifier of the component to be started.
  
  @param behaviour: identifier of the behaviour to be started on test component.
  
  @param parameterList: list of values, each value defines a parameter. The parameters are ordered as described in TTCN-3 declaration. Null or an empty list can be passed if no parameter.
  */

void tciStopTestComponentReq(TriComponentId* component);
  /*
  CH transmits stop component request to the remote TE, and calls tciStopTestComponent.
  
  @param component: identifier of the component to stop.
  */

void tciConnectReq(TriPortId* fromPort, TriPortId* toPort);
  /*
  CH transmits connection request to the remote TE, and calls tciConnect.
  
  @param fromPort: identifer of the test component port to be connected from.
  
  @param toPort: identifer of the test component port to be connected to.
  */

void tciDisconnectReq(TriPortId* fromPort, TriPortId* toPort);
  /*
  CH transmits disconnection request to the remote TE, and calls tciDisconnect.
  
  @param fromPort: identifer of the test component port to be disconnected.
  
  @param toPort: identifer of the test component port to be disconnected.
  */

void tciMapReq(TriPortId* fromPort, TriPortId* toPort);
  /*
  CH transmits map request to the remote TE, and calls tciMap.
  
  @param fromPort: identifer of the test component port to be mapped from.
  
  @param toPort: identifer of the test component port to be mapped to.
  */

void tciUnmapReq(TriPortId* fromPort, TriPortId* toPort);
  /*
  CH transmits unmap request to the remote TE, and calls tciUnmap.

  @param fromPort: identifer of the test component port to be unmapped.
  
  @param toPort: identifer of the test component port to be unmapped.
  */

void tciTestComponentTerminatedReq(TriComponentId* component, VerdictValue verdict);
  /*
  CH is notified of the termination an the indicated component, then communicates this termination to all participating TEs.
  
  @param component: identifier of the terminated component.
  
  @param verdict: verdict after termination.
  */

Boolean tciTestComponentRunningReq(TriComponentId* component);
  /*
  CH transmits running request to the remote TE, where tciTestComponentRunning is called.
  
  @param component: identifier of the component to test.
  
  @return: true if the component is still running, false otherwise.
  */

Boolean tciTestComponentDoneReq(TriComponentId* component);
  /*
  CH transmits done request to the remote TE, where tciTestComponentRunning is called.
  
  @param component: identifier of the component to test.
  
  @return: true if the component has complete execution, false otherwise.
  */

TriComponentId* tciGetMTCReq();
  /*
  CH determines the ID of the MTC
  
  @return: a TriComponentId value of the MTC.
  */

void tciExecuteTestCaseReq(TciTestCaseIdType* testCaseId, TriPortIdList* tsiPortList);
  /*
  CH transmits execute test case request to the remote TE.
  
  @param testCaseId: test case identifier.

  @param tsiPortList: list of all TSI ports if TSi has been definied, MTC ports otherwise. If no port, null value or empty list shall be passed.
  */

void tciResetReq();
  /*
  CH transmits reset request to all involved TEs
  */

void tciKillTestComponentReq(TriComponentId* component);
  /*
  CH transmits kill component request to the remote TE, and calls tciKillTestComponent.
  
  @param component: identifier of the component to kill.
  */

Boolean tciTestComponentAliveReq(TriComponentId* component);
  /*
  CH transmits alive component request to the remote TE, and calls tciTestComponentAlive.
  
  @param component: identifier of the component to test.
  
  @return: true if component is alive, false otherwise.
  */

Boolean tciTestComponentKilledReq(TriComponentId* component);
  /*
  CH transmits killed component request to the remote TE, and calls tcTestComponentKilled.
  
  @param component: identifier of the component to test.
  
  @return: true if component has been killed, false otherwise.
  */

/*********************************************************************************************************************************************/
/*
 *TCI-TL required provided
 *TL presents all information provided in the parameters of these operations to the user
 */
/*********************************************************************************************************************************************/

void tliTcExecute(char* am, int ts, char* src, int line, TriComponentId* c, TciTestCaseIdType* tcId, TriParameterList* triPars, TriTimerDuration dur);
  /*
  Shall be called by TE to log the execute test case request.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param tcId: test case executed.
  
  @param triPars: The list of parameters required by the testcase.
  
  @param dur: Duration of the execution.
  */

void tliTcStart(char* am, int ts, char* src, int line, TriComponentId* c, TciTestCaseIdType* tcId, TriParameterList* tciPars, TriTimerDuration dur);
  /*
  Shall be called by TE to log the start of a test case(before test case is started).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param tcId: test case executed.
  
  @param triPars: The list of parameters required by the testcase.
  
  @param dur: Duration of the execution.
  */

void tliTcStop(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TE to log the stop of a test case.

  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliTcStarted(char* am, int ts, char* src, int line, TriComponentId* c, TciTestCaseIdType* tcId, TciParameterListType* tciPars, TriTimerDuration dur);
  /*
  Shall be called by TM to log the stop of a test case(after test case is started).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param tcId: test case executed.
  
  @param triPars: The list of parameters required by the testcase.
  
  @param dur: Duration of the execution.
  */
  
void tliTcTerminated(char* am, int ts, char* src, int line, TriComponentId* c, TciTestCaseIdType* tcId, TciParameterListType* tciPars, VerdictValue verdict);
  /*
  Shall be called by TM to log the termination of a test case(after the test case terminated).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param tcId: test case executed.
  
  @param triPars: The list of parameters required by the testcase.
  
  @param verdict: the verdict of the test case.
  */

void tliCtrlStart(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TE to log the start of a control part(before control part is started).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliCtrlStop(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TE to log the stop of a control part(before control part is stopped).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliCtrlTerminated(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TM to log the termination of a control part(after control part is terminated).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliMSend_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, Value* msgValue, TriAddress* address, TciStatus* encoderFailure, TriMessage* msg, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a unicast send operation(after sending). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is sent.
  
  @param to: the port to which the message is sent.
  
  @param msgValue: the value to be encoded and sent.
  
  @param address: The address of the destination within the SUT.
  
  @param encoderFailure: The failure message which might occur at encoding.
  
  @param msg: The encoded message.
  
  @param transmissionFailure: The failure message which might occur at transmission.
  */

void tliMSend_m_BC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, Value* msgValue, TciStatus* encoderFailure, TriMessage* msg, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a broadcast send operation(after sending). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is sent.
  
  @param to: the port to which the message is sent.
  
  @param msgValue: the value to be encoded and sent.
  
  @param encoderFailure: The failure message which might occur at encoding.
  
  @param msg: The encoded message.
  
  @param transmissionFailure: The failure message which might occur at transmission.
  */

void tliMSend_m_MC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, Value* msgValue, TriAddressList* addresses, TciStatus* encoderFailure, TriMessage* msg, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a multicast send operation(after sending). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is sent.
  
  @param to: the port to which the message is sent.
  
  @param msgValue: the value to be encoded and sent.
  
  @param addresses: the addresses of the destination within the SUT.
  
  @param encoderFailure: The failure message which might occur at encoding.
  
  @param msg: The encoded message.
  
  @param transmissionFailure: The failure message which might occur at transmission.
  */

void tliMSend_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, Value* msgValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a unicast send operation(after sending). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is sent.
  
  @param to: the port to which the message is sent.
  
  @param msgValue: the value to be encoded and sent.
  
  @param transmissionFailure: The failure message which might occur at transmission.
  */

void tliMSend_c_BC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortIdList* to, Value* msgValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a broadcast send operation(after sending). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is sent.
  
  @param to: the ports to which the message is sent.
  
  @param msgValue: the value to be encoded and sent.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliMSend_c_MC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortIdList* to, Value* msgValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a multicast send operation(after sending). Used for logging the intercomponent communication
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is sent.
  
  @param to: the ports to which the message is sent.
  
  @param msgValue: the value to be encoded and sent.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliMDetected_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* from, TriMessage* msg, TriAddress* address);
  /*
  Shall be called by SA to log the enqueuing of a message(after the message is enqueued). Used for logging the communication with the SUT
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is received.
  
  @param from: the port from which the message has been sent.
  
  @param msgValue: the received encoded message.
  
  @param address: the address of the source within the SUT.
  */

void tliMDetected_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* from, Value* msgValue);
  /*
  Shall be called by CH to log the enqueuing of a message(after the message is enqueued). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is received.
  
  @param from: the port from which the message has been sent.
  
  @param msgValue: the received encoded message.
  */

void tliMMismatch_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, Value* msgValue, TciValueTemplate* msgTmpl, TciValueDifferenceList* diffs, TriAddress* address, TciValueTemplate* addressTmpl);
  /*
  Shall be called by TE to log the mismatch of a template(after checking a template match). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is received.
  
  @param msgValue: the message which is checked against the template.
  
  @param msgTmpl: the used template.
  
  @param diffs: difference between message and template.
  
  @param address: the address of the source within the SUT.
  
  @param addressTmpl: the expected address of the source within the SUT.
  */

void tliMMismatch_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, Value* msgValue, TciValueTemplate* msgTmpl, TciValueDifferenceList* diffs, TriComponentId* from, TciNonValueTemplate* fromTmpl);
  /*
  Shall be called by TE to log the mismatch of a template(after checking a template match). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is received.
  
  @param msgValue: the message which is checked against the template.
  
  @param msgTmpl: the used template.
  
  @param diffs: difference between message and template.
  
  @param from: component which sent the message.
  
  @param fromTmpl: the expected sender component.
  */

void tliMReceive_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, Value* msgValue, TciValueTemplate* msgTmpl, TriAddress* address, TciValueTemplate* addressTmpl);
  /*
  Shall be called by TE to log the receipt of a message(after checking a template match). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is received.
  
  @param msgValue: the message which is checked against the template.
  
  @param msgTmpl: the used template.
  
  @param address: the address of the source within the SUT.
  
  @param addressTmpl: the expected address of the source within the SUT.
  */

void tliMReceive_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, Value* msgValue, TciValueTemplate* msgTmpl, TriComponentId* fromComp, TciNonValueTemplate* fromTmpl);
  /*
  Shall be called by TE to log the receipt of a message(after checking a template match). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the message is received.
  
  @param msgValue: the message which is checked against the template.
  
  @param msgTmpl: the used template.
  
  @param fromComp: component which sent the message.
  
  @param fromTmpl: the expected sender component.
  */

void tliPrCall_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, TriAddress* address, TriStatus encoderFailure, TriParameterList* triPars, TciStatus* transmissionFailure);
  /*
  Shall be called by SA to log a unicast call operation(after call execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is invoked.
  
  @param to: the port to which the call is sent.
  
  @param signature: signature of called procedure.
  
  @param tciPars: parameters of called procedure.
  
  @param address: the address of the destination within the SUT.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param triPars: the encoded parameters.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrCall_m_BC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, TciStatus* encoderFailure, TriParameterList* triPars, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a broadcast call operation(after call execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is invoked.
  
  @param to: the port to which the call is sent.
  
  @param signature: signature of called procedure.
  
  @param tciPars: parameters of called procedure.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param triPars: the encoded parameters.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrCall_m_MC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, TriAddressList* addresses, TciStatus* encoderFailure, TriParameterList* triPars, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a multicast call operation(after call execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is invoked.
  
  @param to: the port to which the call is sent.
  
  @param signature: signature of called procedure.
  
  @param tciPars: parameters of called procedure.
  
  @param addresses: the addresses of the destination within the SUT.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param triPars: the encoded parameters.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrCall_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a unicast call operation(after call execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is invoked.
  
  @param to: the port to which the call is sent.
  
  @param signature: signature of called procedure.
  
  @param tciPars: parameters of called procedure.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrCall_c_BC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortIdList* to, TriSignatureId* signature, TciParameterListType* tciPars, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a broadcast call operation(after call execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is invoked.
  
  @param to: the ports to which the call is sent.
  
  @param signature: signature of called procedure.
  
  @param tciPars: parameters of called procedure.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrCall_c_MC(char* am, int ts, char* src,  int line, TriComponentId* c, TriPortId* at, TriPortIdList* to, TriSignatureId* signature, TciParameterListType* tciPars, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a multicast call operation(after call execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is invoked.
  
  @param to: the ports to which the call is sent.
  
  @param signature: signature of called procedure.
  
  @param tciPars: parameters of called procedure.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrGetCallDetected_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* from, TriSignatureId* signature, TriParameterList* triPars, TriAddress* address);
  /*
  Shall be called by the SA to log the getcall enqueue operation(after call is enqueued). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is received.
  
  @param from: the port from which the call has been sent.
  
  @param signature: signature of detected procedure.
  
  @param triPars: encoded parameters of detected procedure.
  
  @param address: the address of the destination within the SUT.
  */

void tliPrGetCallDetected_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* from, TriSignatureId* signature, TciParameterListType* tciPars);
  /*
  Shall be called by the CH to log the getcall enqueue operation(after call is enqueued). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is received.
  
  @param from: the port from which the call has been sent.
  
  @param signature: signature of detected procedure.
  
  @param triPars: encoded parameters of detected procedure.
  */

void tliPrGetCallMismatch_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, TciParameterListType* tciPars, TciValueTemplate* parsTmpl, TciValueDifferenceList* diffs, TriAddress* address, TciValueTemplate* addressTmpl);
  /*
  Shall be called by TE to log the mismatch of a getcall(after getcall is checked against a template). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is received.
  
  @param signature: signature of detected procedure.
  
  @param tciPars: parameters of detected call.
  
  @param parsTmpl: used template.
  
  @param diffs: differences between parameters and template.
  
  @param address: the address of the source within the SUT.
  
  @param addressTmpl: the expected address of the source within the SUT.
  */

void tliPrGetCallMismatch_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, TciParameterListType* tciPars, TciValueTemplate* parsTmpl, TciValueDifferenceList* diffs, TriComponentId* from, TciNonValueTemplate* fromTmpl);
  /*
  Shall be called by TE to log the mismatch of a getcall(after getcall is checked against a template). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is received.
  
  @param signature: signature of detected procedure.
  
  @param tciPars: parameters of detected call.
  
  @param parsTmpl: used template.
  
  @param diffs: differences between parameters and template.
  
  @param from: component which called the opeartion.
  
  @param fromTmpl: the expected calling component.
  */

void tliPrGetCall_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, TciParameterListType* tciPars, TciValueTemplate* parsTmpl, TriAddress* address, TciValueTemplate* addressTmpl);
  /*
  Shall be called by TE to log getting a call(after getcall has matched against a template). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is received.
  
  @param signature: signature of detected procedure.
  
  @param tciPars: parameters of detected call.
  
  @param parsTmpl: used template.
  
  @param address: the address of the source within the SUT.
  
  @param addressTmpl: the expected address of the source within the SUT.
  */

void tliPrGetCall_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, TciParameterListType* tciPars, TciValueTemplate* parsTmpl, TriComponentId* from, TciNonValueTemplate* fromTmpl);
  /*
  Shall be called by TE to log getting a call(after getcall has matched against a template). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the call is received.
  
  @param signature: signature of detected procedure.
  
  @param tciPars: parameters of detected call.
  
  @param parsTmpl: used template.
  
  @param from: component which called the opeartion.
  
  @param fromTmpl: the expected calling component.
  */

void tliPrReply_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* replValue, TriAddress* address, TciStatus* encoderFailure, TriParameterList* triPars, TriParameter* repl, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a unicast reply operation(after reply execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is sent.
  
  @param to: the port to which the reply is sent.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param replValue: reply to be sent.
  
  @param address: the address of the destination within the SUT.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param triPars: encoded parameters.
  
  @param repl: encoded reply.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrReply_m_BC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* replValue, TciStatus* encoderFailure, TriParameterList* triPars, TriParameter* repl, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a broadcast reply operation(after reply execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is sent.
  
  @param to: the port to which the reply is sent.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param replValue: reply to be sent.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param triPars: encoded parameters.
  
  @param repl: encoded reply.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrReply_m_MC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* replValue, TriAddressList* addresses, TriStatus encoderFailure, TriParameterList* triPars, TriParameter* repl, TciStatus* transmissionFailure);
  /*
  Shall be called by SA to log a multicast reply operation(after reply execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is sent.
  
  @param to: the port to which the reply is sent.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param replValue: reply to be sent.
  
  @param addresses: the addresses of the destination within the SUT.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param triPars: encoded parameters.
  
  @param repl: encoded reply.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrReply_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* replValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a unicast reply operation(after reply execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is sent.
  
  @param to: the port to which the reply is sent.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param replValue: reply to be sent.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrReply_c_BC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortIdList* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* replValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a broadcast reply operation(after reply execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is sent.
  
  @param to: the ports to which the reply is sent.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param replValue: reply to be sent.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrReply_c_MC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortIdList* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* replValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a multicast reply operation(after reply execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is sent.
  
  @param to: the ports to which the reply is sent.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param replValue: reply to be sent.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrGetReplyDetected_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* from, TriSignatureId* signature, TriParameterList* triPars, TriParameter* repl, TriAddress* address);
  /*
  Shall be called by SA to log the getreply enqueue operation(after getreply is enqueued). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is received.
  
  @param from: the port from which the reply has been sent.
  
  @param signature: signature relating to the reply.
  
  @param triPars: encoded parameters od detected reply.
  
  @param repl: received encoded reply.
  
  @param address: the address of the source within the SUT.
  */

void tliPrGetReplyDetected_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* from, TriSignatureId* signature, TciParameterListType* tciPars, Value* replValue);
  /*
  Shall be called by CH to log the getcall enqueue operation(after call is enqueued). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is received.
  
  @param from: the port from which the reply has been sent.
  
  @param signature: signature relating to the reply.
  
  @param triPars: encoded parameters od detected reply.
  
  @param repl: received reply.
  */

void tliPrGetReplyMismatch_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, TciParameterListType* tciPars, TciValueTemplate* parsTmpl, Value* replValue, TciValueTemplate* replyTmpl, TciValueDifferenceList* diffs, TriAddress* address, TciValueTemplate* addressTmpl);
  /*
  Shall be called by TE to log the mismatch of a getreply operation(after getreply is checked against a template). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is received.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param parsTmpl: template used to check parameters match.
  
  @param replValue: received reply.
  
  @param replyTmpl: template used to check reply match.
  
  @param diffs: difference between reply and template.
  
  @param address: the address of the source within the SUT.
  
  @param addressTmpl: the expected address of the source within the SUT.
  */

void tliPrGetReplyMismatch_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, TciParameterListType* tciPars, TciValueTemplate* parsTmpl, Value* replValue, TciValueTemplate* replyTmpl, TciValueDifferenceList* diffs, TriComponentId* from, TciNonValueTemplate* fromTmpl);
  /*
  Shall be called by TE to log the mismatch of a getreply operation(after getreply is checked against a template). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is received.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param parsTmpl: template used to check parameters match.
  
  @param replValue: received reply.
  
  @param replyTmpl: template used to check reply match.
  
  @param diffs: difference between reply and template.
  
  @param from: the component which sent the reply.
  
  @param fromTmpl: the expected component.
  */

void tliPrGetReply_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, TciParameterListType* tciPars, TciValueTemplate* parsTmpl, Value* replValue, TciValueTemplate* replyTmpl, TriAddress* address, TciValueTemplate* addressTmpl);
  /*
  Shall be called by TE to log getting a reply(ater getreply is checked against a template). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is received.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param parsTmpl: template used to check parameters match.
  
  @param replValue: received reply.
  
  @param replyTmpl: template used to check reply match.
  
  @param address: the address of the source within the SUT.
  
  @param addressTmpl: the expected address of the source within the SUT.
  */

void tliPrGetReply_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, TciParameterListType* tciPars, TciValueTemplate* parsTmpl, Value* replValue, TciValueTemplate* replyTmpl, TriComponentId* from, TciNonValueTemplate* fromTmpl);
  /*
  Shall be called by TE to log getting a reply(ater getreply is checked against a template). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the reply is received.
  
  @param signature: signature relating to the reply.
  
  @param tciPars: parameters relating to the reply.
  
  @param parsTmpl: template used to check parameters match.
  
  @param replValue: received reply.
  
  @param replyTmpl: template used to check reply match.
  
  @param from: the component which sent the reply.
  
  @param fromTmpl: the expected component.
  */

void tliPrRaise_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* excValue, TriAddress* address, TciStatus* encoderFailure, TriException* exc, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a unicast raise operation(after reply execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is sent.
  
  @param to: the port to which the exeception is sent.
  
  @param signature: signature relating to the exception.
  
  @param tciPars: parameters relating to the exception.
  
  @param excValue: exception to be sent.
  
  @param address: the address of the destination within the SUT.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param exc: encoded exception.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrRaise_m_BC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* excValue, TciStatus* encoderFailure, TriException* exc, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a broadcast raise operation(after reply execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is sent.
  
  @param to: the port to which the exeception is sent.
  
  @param signature: signature relating to the exception.
  
  @param tciPars: parameters relating to the exception.
  
  @param excValue: exception to be sent.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param exc: encoded exception.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrRaise_m_MC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* excValue, TriAddressList* addresses, TciStatus* encoderFailure, TriException* exc, TriStatus transmissionFailure);
  /*
  Shall be called by SA to log a multicast raise operation(after reply execution). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is sent.
  
  @param to: the port to which the exeception is sent.
  
  @param signature: signature relating to the exception.
  
  @param tciPars: parameters relating to the exception.
  
  @param excValue: exception to be sent.
  
  @param addresses: the addresses of the destination within the SUT.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param exc: encoded exception.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrRaise_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* excValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a unicast raise operation(after reply execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is sent.
  
  @param to: the port to which the exeception is sent.
  
  @param signature: signature relating to the exception.
  
  @param tciPars: parameters relating to the exception.
  
  @param excValue: exception to be sent.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrRaise_c_BC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortIdList* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* excValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a broadcast raise operation(after reply execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is sent.
  
  @param to: the ports to which the exeception is sent.
  
  @param signature: signature relating to the exception.
  
  @param tciPars: parameters relating to the exception.
  
  @param excValue: exception to be sent.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrRaise_c_MC(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortIdList* to, TriSignatureId* signature, TciParameterListType* tciPars, Value* excValue, TriStatus transmissionFailure);
  /*
  Shall be called by CH to log a multicast raise operation(after reply execution). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is sent.
  
  @param to: the ports to which the exeception is sent.
  
  @param signature: signature relating to the exception.
  
  @param tciPars: parameters relating to the exception.
  
  @param excValue: exception to be sent.
  
  @param transmissionFailure: the failure message which might occur at transmission.
  */

void tliPrCatchDetected_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* from, TriSignatureId* signature, TriException* exc, TriAddress* address);
  /*
  Shall be called by SA to log the catch enqueue operation(after catch is enqueued). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is received.
  
  @param from: the port from which the exception has been sent.
  
  @param signature: signature relating to the exception.
  
  @param exc: the catched exception.
  
  @param address: the address of the source within the SUT.
  */

void tliPrCatchDetected_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriPortId* from, TriSignatureId* signature, Value* excValue);
  /*
  Shall be called by CH to log the catch enqueue operation(after catch is enqueued). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is received.
  
  @param from: the port from which the exception has been sent.
  
  @param signature: signature relating to the exception.
  
  @param excValue: the catched exception.
  */

void tliPrCatchMismatch_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, Value* excValue, TciValueTemplate* excTmpl, TciValueDifferenceList* diffs, TriAddress* address, TciValueTemplate* addressTmpl);
  /*
  Shall be called by TE to log the mismatch of a catch operation(after catch is checked against a template). Used for logging the communication with the SUT.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is received.
  
  @param signature: signature relating to the exception.
  
  @param excValue: the received exception.
  
  @param excTmpl: used template.
  
  @param diffs: diferences between exception and template.
  
  @param address: the address of the source within the SUT.
  
  @param addressTmpl: the expected address of the source within the SUT.
  */

void tliPrCatchMismatch_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, Value* excValue, TciValueTemplate* excTmpl, TciValueDifferenceList* diffs, TriComponentId* from, TciNonValueTemplate* fromTmpl);
  /*
  Shall be called by TE to log the mismatch of a catch operation(after catch is checked against a template). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is received.
  
  @param signature: signature relating to the exception.
  
  @param excValue: the received exception.
  
  @param excTmpl: used template.
  
  @param diffs: diferences between exception and template.
  
  @param from: the component which sent the reply.
  
  @param fromTmpl: the expected component.
  */

void tliPrCatch_m(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, Value* excValue, TciValueTemplate* excTmpl, TriAddress* address, TciValueTemplate* addressTmpl);
  /*
  Shall be called by SA to log catching an exception(after catch is checked against a template). Used for logging the communication with the SUT.
  
  Shall be called by TE to log the mismatch of a catch operation(after catch is checked against a template). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is received.
  
  @param signature: signature relating to the exception.
  
  @param excValue: the received exception.
  
  @param excTmpl: used template.
  
  @param address: the address of the source within the SUT.
  
  @param addressTmpl: the expected address of the source within the SUT.
  */

void tliPrCatch_c(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature, Value* excValue, TciValueTemplate* excTmpl, TriComponentId* from, TciNonValueTemplate* fromTmpl);
  /*
  Shall be called by CH to log catching an exception(after catch is checked against a template). Used for logging the intercomponent communication.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is received.
  
  @param signature: signature relating to the exception.
  
  @param excValue: the received exception.
  
  @param excTmpl: used template.
  
  @param from: the component which sent the reply.
  
  @param fromTmpl: the expected component.
  */

void tliPrCatchTimeoutDetected(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature);
  /*
  Shall be called by PA to log the detection of a catch timeout(after the timeout is enqueued).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is received.
  
  @param signature: signature relating to the exception.
  */

void tliPrCatchTimeout(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* at, TriSignatureId* signature);
  /*
  Shall be called by TE to log catching a timeout(after the catch timeout has been performed).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param at: the port via which the exception is received.
  
  @param signature: signature relating to the exception.
  */

void tliCCreate(char* am, int ts, char* src, int line, TriComponentId* c, TriComponentId* comp, char* name, Boolean alive);
  /*
  Shall be called by TE to log the create component operation(after component creation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param comp: component which is created.
  
  @param name: name of this component.
  
  @param alive: true is the component is an alive component.
  */

void tliCStart(char* am, int ts, char* src, int line, TriComponentId* c, TriComponentId* comp, TciBehaviourIdType* name, TciParameterListType* tciPars);
  /*
  Shall be called by TE to log the start component operation(after component start).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param comp: component which is started.
  
  @param name: name of this component.
  
  @param alive: parameters of the started behaviour.
  */

void tliCRunning(char* am, int ts, char* src, int line, TriComponentId* c, TriComponentId* comp, ComponentStatusType status);
  /*
  Shall be called by TE to log the running component operation(after component running).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param comp: the checked component.
  
  @param status: status of this component.
  */

void tliCAlive();
  /*
  Shall be called by TE to log the alive component operation(after component alive).
  */

void tliCStop(char* am, int ts, char* src, int line, TriComponentId* c, TriComponentId* comp);
  /*
  Shall be called by TE to log the stop component operation(after component stop).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param comp: component which is stopped.
  */

void tliCKill();
  /*
  Shall be called by TE to log the kill component operation(after component kill).
  */

void tliCDoneMismatch(char* am, int ts, char* src, int line, TriComponentId* c, TriComponentId* comp, TciNonValueTemplate* compTmpl);
  /*
  Shall be called by TE to log the mismatch of a done component operation(after done is checked against a template).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param comp: first component that did not match.
  
  @param compTmpl: used template.
  */

void tliCDone(char* am, int ts, char* src, int line, TriComponentId* c, TciNonValueTemplate* compTmpl);
  /*
  Shall be called by TE to log the done component operation(after the done operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param compTmpl: used template to check done match.
  */

void tliCKilledMismatch();
  /*
  Shall be called by TE to log the mismatch of a killed component operation(after killed is checked against a template).
  */

void tliCKilled();
  /*
  Shall be called by TE to log the killed component operation(after the killed operation).
  */

void tliCTerminated(char* am, int ts, char* src, int line, TriComponentId* c, VerdictValue verdict);
  /*
  Shall be called by TE to log the termination of a component(after the termination of the component).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param verdict: verdict of the component.
  */

void tliPConnect(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* port1, TriPortId* port2);
  /*
  Shall be called by CH to log the connect operation(after the connect operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param port1: first port to be connected.
  
  @param port2: second port to be connected.
  */

void tliPDisconnect(char* am, int ts, char* src, int line, TriComponentId* c, TriComponentId* c1, TriPortId* port1, TriComponentId* c2, TriPortId* port2);
  /*
  Shall be called by CH to log the disconnect operation(after the disconnect operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param c1:
  
  @param port1: first port to be disconnected.
  
  @param c2:
  
  @param port2: second port to be disconnected.
  */

void tliPMap(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* port1, TriPortId* port2);
  /*
  Shall be called by SA to log the map operation(after the map operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param port1: first port to be mapped.
  
  @param port2: second port to be mapped.
  */

void tliPUnmap(char* am, int ts, char* src, int line, TriComponentId* c, TriComponentId* c1, TriPortId* port1, TriComponentId* c2, TriPortId* port2);
  /*
  Shall be called by SA to log the unmap operation(after the unmap operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param c1:
  
  @param port1: first port to be unmapped.
  
  @param c2:
  
  @param port2: second port to be unmapped.
  */

void tliPClear(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* port);
  /*
  Shall be called by TE to log the port clear operation(after the port clear operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param port: port to be cleared.
  */

void tliPStart(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* port);
  /*
  Shall be called by TE to log the port start operation(after the port start operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param port: port to be started.
  */

void tliPStop(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* port);
  /*
  Shall be called by TE to log the port stop operation(after the port stop operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param port: port to be stopped.
  */

void tliPHalt(char* am, int ts, char* src, int line, TriComponentId* c, TriPortId* port);
  /*
  Shall be called by TE to log the port halt operation(after the port halt operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param port: port to be stopped.
  */

void tliEncode(char* am, int ts, char* src, int line, TriComponentId* c, Value* val, TciStatus* encoderFailure, TriMessage* msg, char* codec);
  /*
  Shall be called by CD to log the encode operation.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param val: value to be encoded.
  
  @param encoderFailure: the failure message which might occur at encoding.
  
  @param msg: encoded value
  
  @param codec: used encoder.
  */

void tliDecode(char* am, int ts, char* src, int line, TriComponentId* c, TriMessage* msg, TciStatus* decoderFailure, Value* val, char* codec);
  /*
  Shall be called by CD to log the decode operation.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param msg: value to be decoded.
  
  @param decoderFailure: the failure message which might occur at decoding.
  
  @param val: decoded value.
  
  @param codec: used decoder.
  */

void tliTTimeoutDetected(char* am, int ts, char* src, int line, TriComponentId* c, TriTimerId* timer);
  /*
  Shall be called by PA to log the detection of a timeout(after timeout is enqueued).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param timer: timer that timed out.
  */

void tliTTimeoutMismatch(char* am, int ts, char* src, int line, TriComponentId* c, TriTimerId* timer, TciNonValueTemplate* timerTmpl);
  /*
  Shall be called by TE to log a timeout mismatch(after a timeout match failed).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param timer: first timer that did not match.
  
  @param timerTmpl: used template.
  */

void tliTTimeout(char* am, int ts, char* src, int line, TriComponentId* c, TriTimerId* timer, TciNonValueTemplate* timerTmpl);
  /*
  Shall be called by TE to log a timeout match(after a timeout matched).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param timer: timer that matched.
  
  @param timerTmpl: used template.
  */

void tliTStart(char* am, int ts, char* src, int line, TriComponentId* c, TriTimerId* timer, TriTimerDuration dur);
  /*
  Shall be called by PA to log the start of a timer(after the start timer operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param timer: timer that is started.
  
  @param dur: its duration.
  */

void tliTStop(char* am, int ts, char* src, int line, TriComponentId* c, TriTimerId* timer, TriTimerDuration dur);
  /*
  Shall be called by PA to log the stop of a timer(after the stop timer operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param timer: timer that is stopped.
  
  @param dur: duration when ot was stopped.
  */

void tliTRead(char* am, int ts, char* src, int line, TriComponentId* c, TriTimerId* timer, TriTimerDuration elapsed2);
  /*
  Shall be called by PA to log the reading of a timer(after the read timer operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param timer: timer that is started.
  
  @param elapsed: elapsed time of the timer.
  */

void tliTRunning(char* am, int ts, char* src, int line, TriComponentId* c, TriTimerId* timer, TimerStatusType status);
  /*
  Shall be called by PA to log the running timer operation(after the running timer operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param timer: timer that is tested.
  
  @param status: status of the timer.
  */

void tliSEnter(char* am, int ts, char* src, int line, TriComponentId* c, QualifiedName* name, TciParameterListType* tciPars, char* kind);
  /*
  Shall be called by TE to log the entering of a scope(after the scope has been entered).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param name: name of the scope.
  
  @param tciPars: parameters of the scope.
  
  @param kind: kind of scope.
  */

void tliSLeave(char* am, int ts, char* src, int line, TriComponentId* c, QualifiedName* name, TciParameterListType* tciPars, Value* returnValue, char* kind);
  /*
  Shall be called by TE to log the leaving of a scope(after the scope has been left).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param name: name of the scope.
  
  @param tciPars: parameters of the scope.
  
  @param returnValue: the returned value of the scope.
  
  @param kind: kind of scope.
  */

void tliVar(char* am, int ts, char* src, int line, TriComponentId* c, QualifiedName* name, Value* varValue);
  /*
  Shall be called by TE to log the modification of the value of a variable(after the values have been changed).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param name: name of the variable.
  
  @param varValue: new value of the variable.
  */

void tliModulePar(char* am, int ts, char* src, int line, TriComponentId* c, QualifiedName* name, Value* parValue);
  /*
  Shall be called by TE to log the value of a module parameter(after the access to the value of a module parameter).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param name: name of the parameter.
  
  @param parValue: new value of the parameter.
  */

void tliGetVerdict(char* am, int ts, char* src, int line, TriComponentId* c, VerdictValue verdict);
  /*
  Shall be called by TE to log the getverdict operation(after the getverdict operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param verdict: value of the current verdict.
  */

void tliSetVerdict(char* am, int ts, char* src, int line, TriComponentId* c, VerdictValue verdict);
  /*
  Shall be called by TE to log the setverdict operation(after the setverdict operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param verdict: value to be set to the current verdict.
  */

void tliLog(char* am, int ts, char* src, int line, TriComponentId* c, char* log);
  /*
  Shall be called by TM to log the TTCN-3 statement log(after the TTCN-3 log operation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param log: string to be logged.
  */

void tliAEnter(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TE to log entering an alt(after an alt has been entered).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliALeave(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TE to log leaving an alt(after the alt has been left).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliANomatch(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TE to log the nomatch of an alt(after the alt has not matched).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliARepeat(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TE to log repeating an alt(when the alt is being repeated).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliADefaults(char* am, int ts, char* src, int line, TriComponentId* c);
  /*.
  Shall be called by TE to log entering the default section(after the default section has been entered)
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliAActivate(char* am, int ts, char* src, int line, TriComponentId* c, QualifiedName* name, TciParameterListType* tciPars, Value* ref);
  /*
  Shall be called by TE to log the activation of a default(after the default activation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param name: name of the default.
  
  @param tciPars: parameters of the default.
  
  @param ref: resulting default reference.
  */

void tliADeactivate(char* am, int ts, char* src, int line, TriComponentId* c, Value* ref);
  /*
  Shall be called by TE to log the deactivation of a default(after the default deactivation).
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param ref: resulting default reference.
  */

void tliAWait(char* am, int ts, char* src, int line, TriComponentId* c);
  /*
  Shall be called by TE to log that the component awaits events for a new snapshot.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  */

void tliAction(char* am, int ts, char* src, int line, TriComponentId* c, char* action);
  /*
  Shall be called by TE to log that the component executed an SUT action.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param action: action to be performed.
  */

void tliMatch(char* am, int ts, char* src, int line, TriComponentId* c, Value* expr, TciValueTemplate* tmpl);
  /*
  Shall be called by TE to log that the component successfully executed a match operation.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param expr: expression to be matched.
  
  @param tmpl: used template.
  */

void tliMatchMismatch(char* am, int ts, char* src, int line, TriComponentId* c, Value* expr, TciValueTemplate* tmpl, TciValueDifferenceList* diffs);
  /*
  Shall be called by TE to log that the component unsuccessfully executed a match operation � a mismatch occurred.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param expr: expression to be matched.
  
  @param tmpl: used template.
  
  @param diffs: difference between expression and template.
  */

void tliInfo(char* am, int ts, char* src, int line, TriComponentId* c, int level, char* info);
  /*
  Can be called by TE to log additional information during test execution.
  
  @param am: additional message.
  
  @param ts: time when the event produced.
  
  @param src: source file of test specification.
  
  @param line: line number.
  
  @param c: component which product this event.
  
  @param level: information level.
  
  @param info: the information.
  */

#ifdef __cplusplus
}
#endif
  
#endif

