/* PragmaDev RTDS Uitron 3 integration */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* RTDS include */
#include "RTDS_Error.h"
#include "RTDS_OS.h"
#include "RTDS_MACRO.h"


/*
 * SEMAPHORE PROTOTYPE:
 * -------------------
 * Variable declared as global to the all the program
 */
RTDS_SYS_TIME_PROTO
RTDS_CRITICAL_SECTION_PROTO
RTDS_START_SYNCHRO_PROTO
RTDS_CRITICAL_TRACE_SECTION_PROTO
extern long RTDS_globalSystemTime;
extern long RTDS_globalStartTime;

/* **************************************************************** *
* If element n of the array is 1 the semaphore ID n is not available, 
* if it is 0 it is available
* **************************************************************** */
unsigned char RTDS_globalSemaphoreId[ RTDS_MAX_SEMAPHORE + CYGNUM_UITRON_SEMAS_INITIALLY];

/* **************************************************************** *
* If element n of the array is 1 the task ID n is not available, 
* if it is 0 it is available
* **************************************************************** */
unsigned char RTDS_globalTaskUniqueId[ RTDS_MAX_TASK + CYGNUM_UITRON_TASKS_INITIALLY];

/* **************************************************************** *
* If element n of the array is 1 the alarm ID n is not available, 
* if it is 0 it is available
* **************************************************************** */
unsigned char RTDS_globalTimerUniqueId[ RTDS_MAX_TIMER + 1];

/* **************************************************************** *
* If element n of the array is 1 the mailboxe ID n is not available, 
* if it is 0 it is available
* **************************************************************** */
unsigned char RTDS_globalMsgBoxUniqueId[ RTDS_MAX_MSGBOX + CYGNUM_UITRON_MBOXES_INITIALLY];


/********************************************************************
 *                    FUNCTION    RTDS_MPoolCreate
 *
 *-------------------------------------------------------------------
 * Macro for using memory pool instead of compiler malloc function
 *
 * PARAMETERS
 *      size : int    size of the memory pool
 *
 *********************************************************************/
void RTDS_MPoolCreate( int size )
    {
    T_CMPL* pk_cmpl = NULL;
    ER    retCode;

    pk_cmpl = ( T_CMPL* )malloc( sizeof( T_CMPL ) );
    if( pk_cmpl == NULL )
        {         
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MALLOC )
        }
    
    memset( pk_cmpl, 0, sizeof( T_CMPL ) );
    
    pk_cmpl->mplsz = size; /* Variable-sized Memory area size (in bytes)*/
    pk_cmpl->mplatr = TA_TFIFO; /* The variable-sized memory pool's wait queue will be in FIFO order */ 

    retCode = cre_mpl( RTDS_MEMPOOL_ID , pk_cmpl );
    if ( retCode != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_ALLOCATING_MEMORY_POOL )
        }
    }

/********************************************************************
 *                    FUNCTION    RTDS_MemAlloc
 *
 *-------------------------------------------------------------------
 * Allocate a block of memory in the variable-sized memory pool (see RTDS_MPoolCreate)
 *
 * PARAMETERS
 *      size : int    size of the memory block
 *
 * RETURN
 *      Pointer on the allocated memory block
 *********************************************************************/
VP RTDS_MemAlloc( int size )
    {
    VP p_blk;
    ER rvalue;
  
    rvalue = get_blk( &p_blk , RTDS_MEMPOOL_ID , size ); /*Parameter: (ID mplid, UINT blksz,VP* P_BLK */
    if(rvalue != E_OK)
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GETBLK )
        }
      
    memset ( p_blk , 0 , size );
    if(p_blk == NULL)
        {         
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MALLOC )
        }
    return p_blk;  
    }

/********************************************************************
 *                    FUNCTION    RTDS_Free
 *
 *-------------------------------------------------------------------
 * Release a previously allocated memory in the variable-sized memory pool (see RTDS_MemAlloc)
 *
 * PARAMETERS
 *      ptrMem : VP     Pointer to the memory block
 *
 *********************************************************************/
void RTDS_Free( VP ptrMem )
    {
    ER rvalue;
    int mem = RTDS_MEMPOOL_ID;

    /* Parameter: ID mplid,VP blk */    
    rvalue = rel_blk( RTDS_MEMPOOL_ID , ptrMem ); 
    if( rvalue != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_RELBLK )
        }
    }

/********************************************************************
 *                                 FUNCTION  cyg_package_start()
 *
 *-------------------------------------------------------------------
 * This function initialize the system
 * ECOSPECIFIC
 ********************************************************************/
externC void cyg_package_start( void )
    {
    cyg_uitron_start();
    RTDS_InitResourceId();
    RTDS_GLOBAL_MEMORY_POOL_CREATION;
    RTDS_CRITICAL_SECTION_INIT;
    }

/********************************************************************
 *                    FUNCTION    RTDS_Sem_Flush
 *
 *-------------------------------------------------------------------
 * Flush a semaphore, allowing threads or process waiting for the semaphore
 * (function parameter) to run
 *
 * PARAMETERS
 *        SemId_To_Flush : RTDS_SemType*    Semaphore ID to flush
 *********************************************************************/
void RTDS_SemFlush( RTDS_SemType *SemId_To_Flush )
    {  
    ER rvalue;

    rvalue = sig_sem( ( ID ) SemId_To_Flush->SemID );
    if( rvalue != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_FLUSH )
        }
    }

/*********************************************************************
 *                                 FUNCTION  RTDS_SemTake
 *
 *--------------------------------------------------------------------
 * Take a semaphore
 *
 * PARAMETERS:
 *        semId : RTDS_SemType*     Semaphore ID to take
 *        time_to_Wait : long           if 0 no wait otherwise wait until semaphore available
 * RETURN:
 *        Function status RTDS_OK if succedded RTDS_ERROR otherwise
 *********************************************************************/
int RTDS_SemTake( RTDS_SemType * semId , long time_to_Wait )
    {
    ER rvalue;

    if( time_to_Wait == RTDS_UITRON_SEM_TIMEOUT_FOREVER )
        {
        rvalue = wai_sem( semId->SemID );
        if( rvalue != E_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_TAKE )
            }
        }
    else
        {
        rvalue = twai_sem( semId->SemID , time_to_Wait );
        if( ( rvalue != E_OK ) && ( rvalue != E_TMOUT ) )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_TAKE )
            }
        }

    if( rvalue == E_OK )
        {
        return RTDS_OK;
        }
    else
        {
        return RTDS_ERROR;
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_SemCCreate
 *
 *-------------------------------------------------------------------
 * Create a counting semaphore
 *
 * PARAMETERS
 *        opt : int                     0:FIFO(TA_TFIFO) or 1:PRIO(TA_TPRIO)
 *        initialState : int         initial state when creating 0: Empty / 1: Full
 * RETURN
 *        Pointer on the created semaphore : SemId
 *********************************************************************/
RTDS_SemType * RTDS_SemCCreate( int opt , int initialState )
    {
    RTDS_SemType *SemId;
    ER rvalue;

    SemId = ( RTDS_SemType * )RTDS_MALLOC( sizeof( RTDS_SemType ) );
    SemId->SemID = ( ID )RTDS_GetResourceId( RTDS_SEM_RESS );
    SemId->RTDS_Info.semType = COUNTING;  
    SemId->RTDS_Info.confStruct.sematr = opt;           /* ATR semaphore attributes TA_TFIFO or TA_TPRI */
    SemId->RTDS_Info.confStruct.isemcnt = initialState; /* INT: initial semaphore count */

    rvalue = cre_sem( ( ID )SemId->SemID , &SemId->RTDS_Info.confStruct );
    if( rvalue != E_OK)
        {
        RTDS_ReleaseResourceId( RTDS_SEM_RESS , SemId->SemID );
        RTDS_FREE( SemId );
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_CREATE ) 
        }
    return SemId;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_SemBCreate
 *
 *-------------------------------------------------------------------
 * Create a binary semaphore
 *
 * PARAMETERS:
 *        opt : int                 0:FIFO(TA_TFIFO) or 1:PRIO(TA_TPRIO)
 *        initialState : int     initial state when creating 0: Empty / 1: Full
 * RETURN:
 *        Pointer on the created semaphore : SemId
 *********************************************************************/
RTDS_SemType * RTDS_SemBCreate( int opt , int initialState )
    {
    RTDS_SemType * SemId;

    SemId = RTDS_SemCCreate( opt , initialState );
    SemId->RTDS_Info.semType = BINARY;

    return ( RTDS_SemType * )SemId;
    }

/********************************************************************
 *                    FUNCTION    RTDS_SemMCreate
 *
 *-------------------------------------------------------------------
 * Create a mutex semaphore
 *
 * RETURN
 *        Pointer on the created semaphore : SemId
 *********************************************************************/
RTDS_SemType * RTDS_SemMCreate()
    {
    RTDS_SemType *SemId;

    SemId = RTDS_SemCCreate( 0 , 1 );
    SemId->RTDS_Info.semType = MUTEX;

    return ( RTDS_SemType * )SemId;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_SemDelete
 *
 *-------------------------------------------------------------------
 * Delete a semaphore
 *
 *
 * PARAMETERS:
 *        semId : RTDS_SemType*    semaphore ID to delete
 *********************************************************************/
void RTDS_SemDelete( RTDS_SemType *semId )
    {
    ER rvalue;
  
    rvalue=del_sem( semId->SemID );
    if( rvalue != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_DEL ) 
        }
    /* Release the ressource ID */
    RTDS_ReleaseResourceId( RTDS_SEM_RESS , semId->SemID );
    RTDS_FREE( semId );
    }

/********************************************************************
 *                                 FUNCTION  RTDS_SemaphoreIdGive
 *
 *-------------------------------------------------------------------
 * Release a semaphore
 *
 * PARAMETERS:
 *        semId : RTDS_SemType*     semaphore ID to release
 *********************************************************************/
void RTDS_SemaphoreIdGive( RTDS_SemType *semId , RTDS_GlobalProcessInfo * currentContext )
  {
  RTDS_SIMULATOR_TRACE( RTDS_semGive , semId , NULL , currentContext );
  RTDS_SemGive( semId );
  }

/********************************************************************
 *                                 FUNCTION  RTDS_SemGive
 *
 *-------------------------------------------------------------------
 * Release a semaphore
 *
 * PARAMETERS:
 *        semId : RTDS_SemType*     semaphore ID to release
 *********************************************************************/
void RTDS_SemGive( RTDS_SemType *semId )
  {
  int rvalue;
  T_RSEM pk_rsem;

  rvalue = sig_sem( semId->SemID );
  if(rvalue != E_OK )
    {
    RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_GIV ) 
    }
  }

/********************************************************************
 *                                 FUNCTION  RTDS_TickGet()
 *
 *-------------------------------------------------------------------
 * Get the system time (global variable RTDS_SystemTime)
 ********************************************************************/
long RTDS_TickGet()
  {
  ER rvalue;
  SYSTIME sysTime;
  
  rvalue = get_tim( &sysTime );
  if( rvalue != E_OK )
    {
    RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_TIM ) 
    }

  return sysTime;
  }

/********************************************************************
 *                                 FUNCTION  RTDS_TickSet()
 *
 *-------------------------------------------------------------------
 * Set the system time
 * therefore this function set the extern variable (RTDS_globalSystemTime)
 * defined in the main function using System Time MACRO (RTDS_MACRO.h)
 *
 ********************************************************************/
void RTDS_TickSet( long timeValue )
    {
    ER rvalue;
    SYSTIME time;
  
    time = timeValue;
    rvalue = set_tim( &time );
    if( rvalue != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SET_TIM ) 
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_InitResourceId()
 *
 *-------------------------------------------------------------------
 * Initialize all ressources/ID available for semaphore, task, mailboxe 
 * and alarm objects.
 ********************************************************************/
void RTDS_InitResourceId( void )
    {
    int i;
    for ( i = 0 ; i < RTDS_MAX_SEMAPHORE ; i++ )
        {
        if ( i < CYGNUM_UITRON_SEMAS_INITIALLY )
            RTDS_globalSemaphoreId[ i ] = 1;
        else
            RTDS_globalSemaphoreId[ i ] = 0;
        }
    
    for ( i = 0 ; i < RTDS_MAX_TASK ; i++ )
        {
        if ( i < CYGNUM_UITRON_TASKS_INITIALLY )
            RTDS_globalTaskUniqueId[ i ] = 1;
        else
            RTDS_globalTaskUniqueId[ i ] = 0;
        }

    for ( i = 0 ; i < RTDS_MAX_MSGBOX ; i++ )
        {
        if ( i < CYGNUM_UITRON_MBOXES_INITIALLY )
            RTDS_globalMsgBoxUniqueId[ i ] = 1;
        else
            RTDS_globalMsgBoxUniqueId[ i ] = 0;
        }

    for ( i = 0 ; i < RTDS_MAX_TIMER ; i++ )
        {  
        if ( i < RTDS_INITIAL_NB_TIMER )
            RTDS_globalTimerUniqueId[ i ] = 1;
        else
            RTDS_globalTimerUniqueId[ i ] = 0;
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_GetResourceId()
 *
 *-------------------------------------------------------------------
 * This function returns an available semaphore, mailboxe, task or 
 * alarm identifier.
 *
 * PARAMETERS
 *      ressourceType : integer    RTDS_TASK_RESS /RTDS_SEM_RESS /RTDS_MSGB_RESS
 * RETURN
 *      A valid identifier or -1
 ********************************************************************/
int RTDS_GetResourceId( int ressourceType )
    {
    int i = 0;
    int maxSize;
    unsigned char *ressourceIdArray;

    switch ( ressourceType )
        {
        case RTDS_TASK_RESS:
            ressourceIdArray = RTDS_globalTaskUniqueId;
            maxSize = RTDS_MAX_TASK+CYGNUM_UITRON_TASKS_INITIALLY;
            break;
    
        case RTDS_MSGB_RESS:
            ressourceIdArray = RTDS_globalMsgBoxUniqueId;
            maxSize = RTDS_MAX_MSGBOX+CYGNUM_UITRON_MBOXES_INITIALLY;
            break;
    
        case RTDS_SEM_RESS:
            ressourceIdArray = RTDS_globalSemaphoreId;
            maxSize = RTDS_MAX_SEMAPHORE+CYGNUM_UITRON_SEMAS_INITIALLY;
            break;
    
        case RTDS_TIMER_RESS:
            ressourceIdArray = RTDS_globalTimerUniqueId;
            maxSize = RTDS_MAX_TIMER+1;
            break;
        }

    while( ( ressourceIdArray[ i ] == 1 ) && ( i < maxSize ) )
        {
        i++;
        }
        
    if ( i != maxSize )
        {
        ressourceIdArray[ i ] = 1;
        return i + 1;
        }
    else
        {
        return -1;
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_ReleaseResourceId()
 *
 *-------------------------------------------------------------------
 * Since UITRON has static number fo ID : this function free
 * ressource depending of the parameters
 *
 * PARAMETERS
 *       ressourceType : int     RTDS_TASK_RESS /RTDS_SEM_RESS /RTDS_MSGB_RESS
 *       ressourceIndex : int    Index to free
 ********************************************************************/
void RTDS_ReleaseResourceId( int ressourceType , int ressourceIndex )
    {
    unsigned char *ressourceIdArray;

    switch ( ressourceType )
        {
        case RTDS_TASK_RESS:
            ressourceIdArray = RTDS_globalTaskUniqueId;
            break;
        
        case RTDS_MSGB_RESS:
            ressourceIdArray = RTDS_globalMsgBoxUniqueId;
            break;
        
        case RTDS_SEM_RESS:
            ressourceIdArray = RTDS_globalSemaphoreId;
            break;
        
        case RTDS_TIMER_RESS:
            ressourceIdArray = RTDS_globalTimerUniqueId;
            break;
        }
    ressourceIdArray[ ressourceIndex - 1 ] = 0; 
    }

/********************************************************************
 *                                 FUNCTION  RTDS_FindAlarmId()
 *
 *-------------------------------------------------------------------
 * This function find the alarm ID defined for RTDS_WatchDogFunction. 
 * To do this we have to find all alarm armed and return ID which has the 
 * smallest time left.
 *
 * RETURN
 *       Alarm ID which has the smallest time left
 ********************************************************************/
int RTDS_FindAlarmId()
    {
    int i ;
    int maxSize;
    unsigned char *ressourceIdArray;
    T_RALM ralm;
    int retval;
    int tmpLfttim;
    int tmpIndex;
  
    ressourceIdArray = RTDS_globalTimerUniqueId;
    maxSize = RTDS_MAX_TIMER;
  
    for ( tmpIndex = 1,i=0 ; i < maxSize; tmpIndex++ )
        {
        if ( tmpIndex + 1 != maxSize )
            {
            if( ressourceIdArray[ tmpIndex ] == 1 )
                {
                retval = ref_alm( &ralm , tmpIndex + 1 ); 
                if( retval != E_OK )
                    {
                    return -1;
                    }
                tmpLfttim = ralm.lfttim;
                break;
                }
            }
        else
            {
            return -1;
            }
        }
  
    for ( i= tmpIndex; i < maxSize; i++ )
        {
        if ( i + 1 != maxSize )
            {
            if( ressourceIdArray[ i ] == 1 )
                {
                retval = ref_alm( &ralm , i + 1 ); 
                if( retval != E_OK )
                    {
                    return -1;
                    }
                if( ralm.lfttim < tmpLfttim )
                    {
                    tmpIndex = i;
                    tmpLfttim = ralm.lfttim;
                    }
                }
            }
        }
    
    ressourceIdArray[ tmpIndex ] = 0;
    return tmpIndex + 1;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_GetCurrentThread()
 *
 *-------------------------------------------------------------------
 * Returns the current thread ID
 *
 * RETURN
 *     ID   ID of the current thread
 ********************************************************************/
ID RTDS_GetCurrentThread()
    {
    ID threadId;
    ER rvalue;
  
    rvalue = get_tid( &threadId );
    if( rvalue != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_TID ) 
        }
    return threadId;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_GetTimerUniqueId()
 *
 *-------------------------------------------------------------------
 * Get an available timer unique id. Is used to set timers
 *
 * PARAMETERS
 *     timerList : RTDS_TimerState*     Points to the first element of
 *                                                      the chained list of RTDS_TimerState
 *
 * RETURN
 *     The unique timer Id
 ********************************************************************
 * Context:
 *     The chained list of RTDS_TimerState is sorted by increasing
 *     unique timer id
 *     timerUniqueId = NULL means the message is not a timer; so it
 *     is avoided.
 ********************************************************************/
long RTDS_GetTimerUniqueId( RTDS_TimerState *timerList )
    {
    RTDS_TimerState *RTDS_prevTimer;
    RTDS_TimerState *RTDS_timer;
    long newTimerId;

    RTDS_CRITICAL_SECTION_START;
    /* If list is empty take 1 */
    if ( timerList == NULL )
        {
        newTimerId = 1;
        }
    /* If 1 is available, take it */
    else if ( timerList->timerUniqueId != 1 )
        {
       newTimerId = 1;
        }
    else
        {
        /* If there is a gap in the timerId chained list, take an Id in the gap */
        RTDS_prevTimer = timerList;
        newTimerId = 0;
        for ( RTDS_timer = timerList->next ; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
            {
            if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 )
                {
                newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                }
            RTDS_prevTimer = RTDS_timer;
            }
        /* No gap, let's take the next value */
        if ( newTimerId == 0 )
            {
            newTimerId = RTDS_prevTimer->timerUniqueId + 1;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
  
    /* Check the counter did not go back to 0 */
    if ( newTimerId == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID );
        }
    return newTimerId;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_WatchDogFunction()
 *
 *-------------------------------------------------------------------
 * Send a timer message using the parameters given to the watchdog
 *
 * PARAMETERS
 *     parameter: RTDS_TimerState*      Needed parameters to create and send the timer:
 *                                                          - QueueId of the receiver
 *                                                          - TimerNumber
 *                                                          - TimerUniqueId
 *                                                          - WatchDogId to destroy it
 * **************************************************************** *
 * Context:
 *     Each time a timer is set, a watchdog is created.
 * **************************************************************** */
void RTDS_WatchDogFunction( RTDS_TimerState *timerState )
    {
    RTDS_TimerState * parameter;
    RTDS_MessageHeader *newMessage;
    ER rvalue;

    #ifdef CYGINT_UITRON_CONFORMANCE
    T_RALM ralm;
    HNO watchDogId = ( HNO )RTDS_FindAlarmId(); 
    rvalue = ref_alm( &ralm , watchDogId );
    if( rvalue != E_OK )
        {
        RTDS_ReleaseResourceId( RTDS_TIMER_RESS , ( int ) watchDogId );
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_START );
        }
    /* We can recover the alarm parameter */
    parameter = ( RTDS_TimerState* )ralm.exinf;
    #else
    parameter = timerState;
    #endif

    newMessage = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );

    /* Fill in the message parameters */
    #ifdef RTDS_SIMULATOR
    newMessage->messageUniqueId = 0;
    #endif
    newMessage->messageNumber = parameter->timerNumber;
    newMessage->timerUniqueId = parameter->timerUniqueId;
    newMessage->sender = parameter->receiverId;
    newMessage->receiver = parameter->receiverId;
    newMessage->next = NULL;
    newMessage->dataLength = 0;
    newMessage->pData = NULL;

    /* Check if process still exist */
    if( RTDS_CheckQueueId( parameter->receiverId ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_QUEUEID );
        }
    else
        {
        rvalue = snd_msg(parameter->receiverId->queueId , ( T_MSG * )newMessage );
        if ( rvalue != E_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG );
            }
        /* Now we can release the ressource ID used for this alarm */
        RTDS_ReleaseResourceId( RTDS_TIMER_RESS , ( int ) watchDogId );
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_forceTimer()
 *
 *-------------------------------------------------------------------
 * Call RTDS_WatchDogFunction()
 * Function called by TCP client to force timer to go off.
 *
 * PARAMETERS
 *         processId : RTDS_ProcessId
 *         timerNumber : int
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
/* By default if RTDS_SIMULATOR is set RTDS_TCP_CLIENT is also */
void RTDS_forceTimer( RTDS_ProcessId myRtosTaskId , int timerNumber )
    {
    RTDS_GlobalProcessInfo *processInfo;
    RTDS_TimerState *timerInfo;
    RTDS_TimerState *timerState;
    long timeOut;
    int status;

    RTDS_CRITICAL_SECTION_START;
    /* Find timerState (RTDS timer ID) from its timer number and the related processId */
    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
        {
        if ( processInfo->myRtosTaskId == myRtosTaskId )
            {
            for ( timerInfo = processInfo->timerList ; timerInfo != NULL ; timerInfo = timerInfo->next )
                {
                if ( timerInfo->timerNumber == timerNumber && timerInfo->state ==RTDS_TIMER_OK )
                    {
                    timerState = timerInfo;
                    timeOut = timerState->timeoutValue;
                    break;
                    }
                }
            }
        }
    RTDS_CRITICAL_SECTION_STOP;

    /* Call timer function to send SDL message */
    if( timerState != NULL )
        {
        /* Let's try to cancel the watchdog */
        status = def_alm( timerState->watchDogId , ( T_DALM * )NADR );
        if( status != E_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_START );
            }
        RTDS_WatchDogFunction( timerState );
        #ifndef RTDS_RT_TIMER
        RTDS_TickSet( timeOut );
        #endif /* RTDS_RT_TIMER */
        }
    }
#endif /* RTDS_TCP_CLIENT */

/********************************************************************
 *                                 FUNCTION  RTDS_StartTimer()
 *
 *-------------------------------------------------------------------
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 *
 * PARAMETERS
 *         - QueueId of the receiver
 *         - TimerNumber
 *         - TimerUniqueId
 *         - delay of the timer
 *         - timerStateList: Address of the list of timer
 * **************************************************************** */
void RTDS_StartTimer( 
 RTDS_SdlInstanceId * queueId , 
 long timerNumber , 
 long timerUniqueId , 
 int delay , 
 RTDS_TimerState **pTimerStateList , 
 RTDS_GlobalProcessInfo * currentContext 
 )
    {
    RTDS_TimerId watchDogId;
    RTDS_TimerState *timerState;
    RTDS_TimerState *previousTimerState;
    RTDS_TimerState *newTimerState;
    int rvalue = RTDS_ERROR;
    T_DALM *configTimer = NULL;

    /* Update the list of timers */
    /* Important note: it is a sorted list based on the timerUniqueId field */
    /* We don't have to test if allocation return a null pointer see RTDS_MemAlloc */
    newTimerState = ( RTDS_TimerState * )RTDS_MALLOC( sizeof( RTDS_TimerState ) );
    watchDogId = ( HNO )RTDS_GetResourceId( RTDS_TIMER_RESS );

    /* Initialize the new element */
    newTimerState->state = RTDS_TIMER_OK;
    newTimerState->timerNumber = timerNumber;
    newTimerState->timeoutValue = delay + RTDS_TickGet() - RTDS_globalStartTime;
    newTimerState->timerUniqueId = timerUniqueId;
    newTimerState->next = NULL;
    newTimerState->watchDogId = watchDogId;
    newTimerState->receiverId = queueId;
      
  
    /* Insert the new element */
    if ( *pTimerStateList == NULL ) /* The list is empty */
        {
        *pTimerStateList = newTimerState;
        }
    else    /* The list is not empty */
        {
        previousTimerState = NULL;
        for ( timerState = *pTimerStateList ; timerState != NULL ; timerState = timerState->next )
            {
            if ( timerState->timerUniqueId > timerUniqueId )
                {
                if ( previousTimerState == NULL )
                    {
                    *pTimerStateList = newTimerState;
                    newTimerState->next = timerState;
                    }
                else
                    {
                    previousTimerState->next = newTimerState;
                    newTimerState->next = timerState;
                    }
                break;
                }
            previousTimerState = timerState;
            }

        if ( timerState == NULL ) /* Inserted at the end of the list */
            {
            previousTimerState->next = newTimerState;
            }
        }

    RTDS_SIMULATOR_TRACE( RTDS_timerStarted, newTimerState , delay , currentContext );

    /* We don't have to test if allocation return a null pointer see RTDS_MemAlloc */
    configTimer = ( T_DALM * )RTDS_MALLOC( sizeof( T_DALM ) );
    configTimer->exinf = newTimerState;  
    configTimer->almatr = ( ATR )TA_HLNG;               /* alarm handler attributes */
    configTimer->almhdr = ( FP )RTDS_WatchDogFunction;  /* alarm handler address */
    configTimer->tmmode = TTM_REL;                      /* start time specification mode */
    configTimer->almtim = delay;                        /* handler startup time */

    /* We need to keep a pointer to this structure in order to release previously allocated memory */
    newTimerState->alarmControl = configTimer;

    rvalue = def_alm( watchDogId , configTimer );
    if( rvalue != E_OK )
        {
        RTDS_ReleaseResourceId( RTDS_TIMER_RESS , ( int ) watchDogId );
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_START );
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_StopTimer()
 *
 *-------------------------------------------------------------------
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * PARAMETERS
 *         - TimerNumber
 *         - pTimerStateList pointing to the timer chained list
 * **************************************************************** */
void RTDS_StopTimer( long timerNumber , RTDS_TimerState **pTimerStateList , RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState *RTDS_timer;
    RTDS_TimerState *RTDS_prevTimer;
    RTDS_MessageHeader *RTDS_message;
    RTDS_MessageHeader *RTDS_prevMessage;
    RTDS_prevTimer = NULL;

    for ( RTDS_timer = *pTimerStateList; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
        {
        if ( ( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ) )
            {
            /* We can release the ressource ID */  
            RTDS_ReleaseResourceId( RTDS_TIMER_RESS , RTDS_timer->watchDogId );
            
            RTDS_SIMULATOR_TRACE( RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext );
            /* Let's try to cancel the watchdog */
            if ( def_alm( RTDS_timer->watchDogId , ( T_DALM * )NADR ) == E_OK )
                {
                /* Remove it from the list */
                /* Is the first of the list */
                if( RTDS_prevTimer == NULL )
                    {
                    *pTimerStateList = RTDS_timer->next;
                    }
                else    /* Not the first of the list */
                    {
                    RTDS_prevTimer->next = RTDS_timer->next;
                    }
                RTDS_FREE( RTDS_timer->alarmControl );
                RTDS_FREE( RTDS_timer );  
                }
                /* Could not cancel the timer. Probably went off allready */
                /* Set it cancelled in the list */
            else
                {
                RTDS_timer->state = RTDS_TIMER_CANCELLED;
                }
            return;
            }
        RTDS_prevTimer = RTDS_timer;
       }

    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->readSaveQueue ; RTDS_message != NULL ; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->readSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            RTDS_FREE( RTDS_message );
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }

    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->writeSaveQueue ; RTDS_message != NULL ; RTDS_message = RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->writeSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            RTDS_FREE( RTDS_message );
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_GetProcessQueueId()
 *
 *-------------------------------------------------------------------
 * Returns the queueId of process based on its name
 *
 * PARAMETERS
 *      processNumber : int      process name as a number defined in RTDS_gen.h
 * RETURN
 *      the process queue id if found if not system error
 ********************************************************************/
RTDS_SdlInstanceId * RTDS_GetProcessQueueId( int processNumber )
    {
    RTDS_GlobalProcessInfo *processInfo;
    RTDS_SdlInstanceId * foundQueue= NULL;

    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
        {
        if ( processInfo->sdlProcessNumber == processNumber )
            {
            foundQueue = processInfo->mySdlInstanceId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;

    if ( foundQueue == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_PROCESS_QUEUE_ID );
        }
    return foundQueue;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_CheckQueueId()
 *
 *-------------------------------------------------------------------
 * Returns RTDS_OK is queueId exist or ERROR if not
 *
 * PARAMETERS
 *      queueId : RTDS_QueueId      Queue Id
 * RETURN
 *      RTDS_OK/ERROR
 * **************************************************************** */
int RTDS_CheckQueueId( RTDS_SdlInstanceId * queueId )
    {
    RTDS_GlobalProcessInfo *processInfo;

    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
        {
        if ( processInfo->mySdlInstanceId == queueId )
            {
            RTDS_CRITICAL_SECTION_STOP;
            return RTDS_OK;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
   
    return RTDS_ERROR;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_MsgQueueSend()
 *
 *-------------------------------------------------------------------
 * Send a message in a process's queue
 *
 * PARAMETERS
 *      messageNumber : long                                  Representing a message name
 *      dataLength : long                                        Length of data pointed by pData
 *      pData : unsigned char*                                Pointer on data sent with the message
 *      receiver  : RTDS_QueueId                          Message receiver queue address
 *      sender : RTDS_QueueId                             Message sender queue address
 *      currentContext : RTDS_GlobalProcessInfo   Current context of the process
 * **************************************************************** */
void RTDS_MsgQueueSend(
 long messageNumber ,
 long dataLength ,
 unsigned char *pData ,
 RTDS_SdlInstanceId * receiver ,
 RTDS_SdlInstanceId * sender ,
 RTDS_GlobalProcessInfo * currentContext 
 )
    {
    ER rvalue;
    RTDS_MessageHeader * RTDS_ptrMessageToSend;
    
    RTDS_ptrMessageToSend = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );
    RTDS_ptrMessageToSend->messageNumber = messageNumber;
    RTDS_ptrMessageToSend->timerUniqueId = 0;
    RTDS_ptrMessageToSend->sender = sender;
    RTDS_ptrMessageToSend->receiver = receiver;
    RTDS_ptrMessageToSend->dataLength = dataLength;
    RTDS_ptrMessageToSend->pData = pData;
    RTDS_ptrMessageToSend->next = NULL;

    #ifdef RTDS_SIMULATOR
    RTDS_ptrMessageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
    #endif

    /* Check if process still exist */
    if( RTDS_CheckQueueId( receiver) == RTDS_OK )
        {
        rvalue = snd_msg( receiver->queueId, ( T_MSG * )RTDS_ptrMessageToSend );
        if ( rvalue != E_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
            }
        RTDS_SIMULATOR_TRACE( RTDS_messageSent , RTDS_ptrMessageToSend , receiver , currentContext );
        }
        
    }
#ifdef RTDS_SIMULATOR

/********************************************************************
 *                                 FUNCTION  RTDS_SimulatorMsgQueueSend()
 *
 *-------------------------------------------------------------------
 * Send a message in a process's queue from the simulator
 * Basically the same as RTDS_MsgQueueSend but it gets the
 * MsgUniqueId as a parameter and does not call the dummy trace
 * function
 *
 * PARAMETERS
 *      messageNumber : long                                  Representing a message name
 *      dataLength : long                                        Length of data pointed by pData
 *      pData : unsigned char*                                Pointer on data sent with the message
 *      receiver  : RTDS_QueueId                          Message receiver queue address
 *      sender : RTDS_QueueId                             Message sender queue address
 *      currentContext : RTDS_GlobalProcessInfo   Current context of the process
 ********************************************************************/
void RTDS_SimulatorMsgQueueSend(
 long messageNumber ,
 long dataLength ,
 unsigned char *pData ,
 RTDS_SdlInstanceId * receiver ,
 RTDS_SdlInstanceId * sender ,
 RTDS_GlobalProcessInfo * currentContext )
    {
    RTDS_MessageHeader *RTDS_ptrMessageToSend;
    ER rvalue;
  
    RTDS_ptrMessageToSend = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );
  
    RTDS_ptrMessageToSend->messageNumber = messageNumber;
    RTDS_ptrMessageToSend->timerUniqueId = 0;
    RTDS_ptrMessageToSend->sender = sender;
    RTDS_ptrMessageToSend->dataLength = dataLength;
    RTDS_ptrMessageToSend->pData = pData;
    RTDS_ptrMessageToSend->next = NULL;
    RTDS_ptrMessageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
  
    rvalue = snd_msg( (RTDS_RtosQueueId)receiver->queueId, ( T_MSG * )RTDS_ptrMessageToSend );
    if ( rvalue != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }
#endif

/********************************************************************
 *                                 FUNCTION  RTDS_MsgQueueReceive()
 *
 *-------------------------------------------------------------------
 * Receives a message from a process's queue
 *
 * PARAMETERS
 *      queueId : RTDS_QueueId      Queue id to read the messsage from
 *      ppMessage : char **              Pointer on the message pointer to receive
 *      size : int                             Size of message
 *      mode : int                            Mode used to read (blocking or non blocking)
 ********************************************************************/
void RTDS_MsgQueueReceive(  RTDS_SdlInstanceId * instanceId,  RTDS_MessageHeader **   ppMessage , int size , int mode)
    {
    ER rvalue = 0;
    rvalue = rcv_msg( ( T_MSG ** )ppMessage , instanceId->queueId);
    if ( rvalue != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_RECEIVE );
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_ProcessCreate()
 *
 *-------------------------------------------------------------------
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 *
 * PARAMETERS
 *      processNumber : int                            Name of the process as a number
 *      functionAddress : void*                        Address of the process function
 *      priority : int                                      Priority fo the process
 *      pOffspringQueueId : RTDS_QueueId*  The address of the address of the msg queue to be created
 *      selfQueueId : RTDS_QueueId              The address of the msg queue of the parent process
 * RETURN
 *      Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */
void RTDS_ProcessCreate(
 int processNumber ,
 void * functionAddress ,
 int priority ,
 RTDS_SdlInstanceId         **pOffspringId,
 RTDS_SdlInstanceId         * selfId,
 short synchronization ,
 RTDS_GlobalProcessInfo * currentContext 
 )
    {
    RTDS_GlobalProcessInfo *processInfo = NULL;
    RTDS_GlobalProcessInfo *newProcessInfo = NULL;
    RTDS_GlobalProcessInfo *previousProcessInfo = NULL;
    RTDS_RtosQueueId mbxId = 0;
    T_CTSK *confStruct = NULL;
    T_CMBX *pk_cmbx = NULL;
    ER rvalue;
  T_CMBX * pk_test;
        previousProcessInfo = NULL;
  
    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
        {
        if ( processInfo->taskstt == 0 )
            {
            if ( previousProcessInfo == NULL )	/* First one in the list */
				         RTDS_globalProcessInfo = processInfo->next;
			      else
			           previousProcessInfo->next = processInfo->next;
			           rvalue=del_tsk(processInfo->myRtosTaskId);
			           if (rvalue!=E_OK)
			               {
			               RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_DELETE )
                     }
			           RTDS_FREE(processInfo);
                 }
        else
        previousProcessInfo = processInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
 
    /* Allocate and fill in a new processInfo structure */
    /* We don't have to test if allocation return a null pointer see RTDS_MemAlloc */
    newProcessInfo = ( RTDS_GlobalProcessInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalProcessInfo ) );

    /* Sets the process name as a number */
    newProcessInfo->sdlProcessNumber = processNumber;
    /* SDL initial state */
    newProcessInfo->sdlState = 0;
    /* Set SDL task priority */
    newProcessInfo->priority = priority;

    /* Last structure of the chained list */
    newProcessInfo->next = NULL;
    newProcessInfo->parentSdlInstanceId = currentContext->mySdlInstanceId;
    newProcessInfo->offspringSdlInstanceId = NULL;
    newProcessInfo->currentMessage = NULL;
    newProcessInfo->timerList = NULL;

    newProcessInfo->mySdlInstanceId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );

    /* New task's queue creation */
    /* We don't have to test if allocation return a null pointer see RTDS_MemAlloc */
    pk_cmbx = ( T_CMBX * )RTDS_MALLOC( sizeof( T_CMBX ) );
    pk_cmbx->mbxatr = TA_TFIFO;
    pk_test = pk_cmbx;
		newProcessInfo->taskQueue = pk_cmbx;
		currentContext->offspringSdlInstanceId = newProcessInfo->mySdlInstanceId;


    mbxId = (RTDS_RtosQueueId  )RTDS_GetResourceId( RTDS_MSGB_RESS );
  
    rvalue = cre_mbx( mbxId , pk_cmbx );
    if( rvalue != E_OK )
        {
        RTDS_ReleaseResourceId( RTDS_MSGB_RESS , ( int )mbxId );
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MBX_CREATE )
        }
    newProcessInfo->mySdlInstanceId->queueId = mbxId;
    newProcessInfo->taskQueue = pk_cmbx;




    /* Take the synchronization semaphore so that the created task does not run straigh away */
    if ( synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_HOLD;
        }
    /* Task creation */
    /* We don't have to test if allocation return a null pointer see RTDS_MemAlloc */
    confStruct = ( T_CTSK * )RTDS_MALLOC( sizeof( T_CTSK ) );
    confStruct->task = ( FP )functionAddress;       /* task task start address        */
    confStruct->itskpri = priority;                 /* PRI initial task priority    */
    confStruct->stksz = RTDS_TASK_STACK_SIZE;       /* INT stack size                 */

    newProcessInfo->myRtosTaskId = ( ID )RTDS_GetResourceId( RTDS_TASK_RESS );
  
    rvalue = cre_tsk( newProcessInfo->myRtosTaskId , confStruct );
    if( rvalue != E_OK )
        {
        RTDS_ReleaseResourceId( RTDS_TASK_RESS , newProcessInfo->myRtosTaskId );
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE )
        }
    newProcessInfo->taskStruct = confStruct;
    newProcessInfo->taskstt = 1;
    /* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalProcessInfo == NULL )
        {
        RTDS_globalProcessInfo = newProcessInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        processInfo = RTDS_globalProcessInfo;
        while( processInfo->next != NULL ) 
            {
            processInfo = processInfo->next;
            }
        processInfo->next = newProcessInfo;
        }
    
    /* Start the created task */
    rvalue = sta_tsk( newProcessInfo->myRtosTaskId , newProcessInfo->myRtosTaskId );
    if( rvalue != E_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_START_TASK );
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_processCreated , newProcessInfo , NULL , currentContext );

    /* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
    if ( synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_UNHOLD;
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_ProcessKill()
 *
 *-------------------------------------------------------------------
 * Kills an SDL process and delete its queue and process info block
 *
 * PARAMETERS
 *      RTDS_currentContext : RTDS_GlobalProcessInfo*   The address of the context of the task to delete
 ********************************************************************/

void RTDS_ProcessKill( RTDS_GlobalProcessInfo *RTDS_currentContext )
    {
    RTDS_GlobalProcessInfo *processInfo;
    RTDS_GlobalProcessInfo *previousProcessInfo;
    RTDS_GlobalProcessInfo *ProcessInfoToKill;

    RTDS_TimerState *RTDS_timer;
    RTDS_MessageHeader *message;
    RTDS_MessageHeader *tmpMessage;
    RTDS_RtosTaskId Queue_toDelete;
    int pidToKill=0;
    ER rvalue;

    /* Let's free the current message if any */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        #ifdef RTDS_SIMULATOR
        /* Release the message unique id back to the pool */
        RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        }
  
    /* Let's clean the timer chained list to free memory and stop watchdogs */
    for ( RTDS_timer = RTDS_currentContext->timerList ; RTDS_timer != NULL ; RTDS_timer = RTDS_currentContext->timerList )
        {
        rvalue = def_alm( RTDS_timer->watchDogId , ( T_DALM * )NADR );
        if( rvalue != E_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_DEF_ALM );
            }
        RTDS_currentContext->timerList = RTDS_timer->next;
        /* Then release the ressource ID and release control block */
        RTDS_ReleaseResourceId( RTDS_TIMER_RESS , RTDS_timer->watchDogId );
        RTDS_FREE( RTDS_timer->alarmControl );
        RTDS_FREE( RTDS_timer );
        }
  
    /* Clean the save queue: free messages and message unique ids in the read and write save queues */
    for ( message = RTDS_currentContext->readSaveQueue ; message != NULL; )
        {
        #ifdef RTDS_SIMULATOR
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
        #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        }
  
    for (message = RTDS_currentContext->writeSaveQueue ; message != NULL; )
        {
        #ifdef RTDS_SIMULATOR
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
        #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        }

    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo ; processInfo != NULL ; processInfo = processInfo->next )
        {
        /* The queue id is used to find the process id */
        if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId  )    /* Found ! */
            {
            RTDS_SIMULATOR_TRACE( RTDS_processDied , processInfo , NULL, RTDS_currentContext );
            /* Update the process information chained list */
                
            pidToKill = processInfo->myRtosTaskId;
            /*Delete the message queue and free the process info block */
            Queue_toDelete=processInfo->mySdlInstanceId->queueId;
            RTDS_ReleaseResourceId( RTDS_MSGB_RESS , ( int )Queue_toDelete );
            RTDS_FREE( processInfo->taskQueue );
            if ( del_mbx( Queue_toDelete ) != E_OK )
                {
                RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_DELETE );
                }
            RTDS_FREE( processInfo->taskStruct );
            processInfo->taskstt=0;
            processInfo->sdlState = -1;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;

    /* Delete the task */
    if ( pidToKill != 0 )
        {
        /* Before exit from this task. Release the ressource ID */
        RTDS_ReleaseResourceId( RTDS_TASK_RESS , pidToKill );
        ext_tsk();
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND );
        }
    }

/********************************************************************
 *                                 FUNCTION  RTDS_Sem_Info_Insert()
 *
 *-------------------------------------------------------------------
 * Inserts a semaphoreId struct in the semaphore info chained list
 *
 * PARAMETERS
 *      semaphoreNumber : int                                   Semaphore number
 *      semaphoreId : RTDS_SemaphoreId                  Id of the semaphore
 *      currentContext : RTDS_GlobalProcessInfo*    Pointer to the context of the task
  * RETURN
 *      RTDS_SemaphoreId    ID of the semaphore inserted
 ********************************************************************/
RTDS_SemaphoreId RTDS_Sem_Info_Insert(
 int semaphoreNumber ,
 RTDS_SemaphoreId semaphoreId ,
 RTDS_GlobalProcessInfo * currentContext
 )
    {
    RTDS_GlobalSemaphoreInfo *semInfo;
    RTDS_GlobalSemaphoreInfo *newSemInfo;

    /* Allocate and fill in a new semInfo structure */
    newSemInfo = ( RTDS_GlobalSemaphoreInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalSemaphoreInfo ) );
    if ( newSemInfo == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT_MALLOC );
        }

    newSemInfo->semaphoreNumber = semaphoreNumber;
    newSemInfo->next = NULL;

    /* Semaphore creation */
    newSemInfo->semaphoreId = semaphoreId;
    if ( newSemInfo->semaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT );
        }
    
    /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalSemaphoreInfo == NULL )
        {
        RTDS_globalSemaphoreInfo = newSemInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        semInfo = RTDS_globalSemaphoreInfo;
        while( semInfo->next != NULL )
            {
            semInfo = semInfo->next;
            }
        semInfo->next = newSemInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;

    RTDS_SIMULATOR_TRACE( RTDS_semaphoreCreated , semaphoreId , -1 , currentContext );

    return semaphoreId;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_SemaphoreIdDelete()
 *
 *-------------------------------------------------------------------
 * Kills a semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 *
 * PARAMETERS
 *      semaphoreId : RTDS_SemaphoreId                 ID of the smeaphore to delete
 *      currentContext : RTDS_GlobalProcessInfo*    Pointer to the current context of the task
 ********************************************************************/
void RTDS_SemaphoreIdDelete( RTDS_SemaphoreId semaphoreId , RTDS_GlobalProcessInfo * currentContext )
    {
    RTDS_GlobalSemaphoreInfo *semInfo;
    RTDS_GlobalSemaphoreInfo *semInfoPrev;

    RTDS_SIMULATOR_TRACE( RTDS_semaphoreDeleted , semaphoreId , 0 , currentContext );
    
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreId == semaphoreId ) /* semaphore found */
            {
            if ( semInfoPrev == NULL )
                {
                RTDS_globalSemaphoreInfo = semInfo->next;
                }
            else
                {
                semInfoPrev->next = semInfo->next;
                }
            RTDS_SemDelete( semInfo->semaphoreId );
            break;
            }
        semInfoPrev = semInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_SemaphoreIdTake()
 *
 *-------------------------------------------------------------------
 * Attempt to take a semaphore from its id
 *
 * PARAMETERS
 *      semaphoreId : RTDS_SemaphoreId               The id of the semaphore
 *      TIME_OUT : RTDS_SemaphoreTimeout             Timeout value
 *      currentContext : RTDS_GlobalProcessInfo*     Pointer to the current context of the task
 * RETURN
 *      RTDS_SemaphoreStatus    RTDS_OK if succedded RTDS_ERROR otherwise
 * **************************************************************** */
RTDS_SemaphoreStatus RTDS_SemaphoreIdTake(
 RTDS_SemaphoreId semaphoreId ,
 RTDS_SemaphoreTimeout TIME_OUT ,
 RTDS_GlobalProcessInfo * currentContext
 )
    {
    RTDS_SemaphoreStatus tmpStatus;

    RTDS_SIMULATOR_TRACE( RTDS_semTakeAttempt , semaphoreId , TIME_OUT , currentContext );
    tmpStatus = RTDS_SemTake( semaphoreId, TIME_OUT );
    if ( tmpStatus == RTDS_OK )
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeSucceded , ( void * )semaphoreId , -1 , currentContext );
        }
    else
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeTimedOut , semaphoreId , 0 , currentContext );
        }
    return tmpStatus;
    }

/********************************************************************
 *                                 FUNCTION  RTDS_GetSemaphoreId()
 *
 *-------------------------------------------------------------------
 * Gets the id of a semaphore from its number (name)
 *
 * PARAMETERS
 *      semaphoreNumber : int       Semaphore number representing its name
 * RETURN
 *      RTDS_SemaphoreId           The id of the semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
    {
    RTDS_GlobalSemaphoreInfo *semInfo;
    RTDS_SemaphoreId foundSemaphoreId=NULL;

    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo ; semInfo != NULL ; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreNumber == semaphoreNumber ) /* semaphore found */
            {
            foundSemaphoreId = semInfo->semaphoreId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;

    if ( foundSemaphoreId==NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_SEMAPHORE_ID );
        }
        
    return foundSemaphoreId;    /* Might cause an RTOS exception if NULL*/
    }

/********************************************************************
 *                 FUNCTION  RTDS_TransitionCleanUp()
 *
 *-------------------------------------------------------------------
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 *
 * PARAMETERS
 *      RTDS_currentContext : RTDS_GlobalProcessInfo*       Pointer to the current context of the task
 *      RTDS_sdlStatePrev : int                                           Previous state of the task
 * **************************************************************** */
void RTDS_TransitionCleanUp( RTDS_GlobalProcessInfo *RTDS_currentContext , int RTDS_sdlStatePrev )
    {
    RTDS_MessageHeader *message = NULL;

    /* Free message buffer if valid */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        #ifdef RTDS_SIMULATOR
        /* Release the message unique id back to the pool */
        RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        RTDS_currentContext->currentMessage = NULL;
        } /* End of if ( RTDS_currentContext->currentMessage != NULL ) */

    /* If SDL state has changed and messages have been saved: reorganise the save queue */
    if ( ( RTDS_currentContext->sdlState != RTDS_sdlStatePrev ) && ( RTDS_currentContext->writeSaveQueue != NULL ) )
        {
        message = RTDS_currentContext->writeSaveQueue;
        
        /* Let's get to the end of the save queue */
        while( message->next != NULL )
            {
            message = message->next;
            }
        
        message->next = RTDS_currentContext->readSaveQueue;
        RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
        RTDS_currentContext->writeSaveQueue = NULL;
        }
    }

/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
char                        *RTDS_globalPrintableParameters=NULL;
void                        *RTDS_globalMessageUniqueIdPool=NULL;
RTDS_GlobalTraceInfo        RTDS_globalTraceEntry = { RTDS_systemError , NULL , 0 , NULL };
#endif

/********************************************************************
 *                                 FUNCTION  RTDS_GetMessageUniqueId()
 *
 *-------------------------------------------------------------------
 * Gets a message unique id for the simulator
 *
 * RETURN
 *      long    The message unique id (minimum is 1)
 * **************************************************************** */
#if defined( RTDS_SIMULATOR )
unsigned long RTDS_GetMessageUniqueId( void )
    {
    unsigned char *index;
    long uniqueByteId;
    long uniqueBitId;

    index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;

    RTDS_CRITICAL_SECTION_START;
    for ( uniqueByteId = 0 ; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE ; uniqueByteId++ )
        {
        if ( *index != 0xFF )
            {
            for ( uniqueBitId = 0 ; uniqueBitId < 8 ; uniqueBitId ++ )
                {
                if ( ( ( 1 << uniqueBitId ) & *index ) ==0 )
                    {
                    *index = *index | ( 1 << uniqueBitId );
                    RTDS_CRITICAL_SECTION_STOP;
                    return ( 8 * uniqueByteId + uniqueBitId + 1 );
                    }
                }
            }
        index++;
        }
    RTDS_CRITICAL_SECTION_STOP;

    /* All bits are set... No more message unique id */
    RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID );
    return 0;
    }
#endif /* RTDS_SIMULATOR */

/********************************************************************
 *                                 FUNCTION  RTDS_ReleaseMessageUniqueId()
 *
 *-------------------------------------------------------------------
 * Make a message unique id available from the pool
 *
 * PARAMETERS
 *      messageUniqueId : unsigned long     Message unique id
 * **************************************************************** */
#if defined( RTDS_SIMULATOR )
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
    {
    unsigned char *index;

    if ( messageUniqueId == 0 ) /* probably a timer */
        {
        return;
        }
    messageUniqueId -= 1;
    index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;
    index += ( unsigned char )( messageUniqueId / 8 );
    RTDS_CRITICAL_SECTION_START;
    ( *index ) = ( *index ) ^ ( 1 << messageUniqueId % 8 );
    RTDS_CRITICAL_SECTION_STOP;
    }
#endif /* RTDS_SIMULATOR */

/********************************************************************
 *                                 FUNCTION  RTDS_GetSystemTime()
 *
 *-------------------------------------------------------------------
 * Get the system time
 *
 * RETURN
 *     unsigned long    System tick count value
 ********************************************************************/
#ifdef RTDS_SIMULATOR
unsigned long RTDS_GetSystemTime( void )
    {
    return RTDS_TickGet();
    }
#endif /* RTDS_SIMULATOR*/

/********************************************************************
 *                                 FUNCTION  RTDS_DummyTraceFunction()
 *
 *-------------------------------------------------------------------
 * The simulator sets a breakpoint on this function and reads 
 * the RTDS_globalTraceEntry variable to see what happened
 *
 ********************************************************************/
#ifdef RTDS_SIMULATOR
void RTDS_DummyTraceFunction( void )
    {
    #ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
    RTDS_TraceAdd();
    #endif
    }
#endif /* RTDS_SIMULATOR */
