/* PragmaDev RTDS uITRON 4 integration */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* RTDS include */
#include "RTDS_OS.h"
#include "RTDS_MACRO.h"

int globalTimeValue = 0;
extern long RTDS_globalSystemTime;
extern long RTDS_globalStartTime;
ER_ID RTDS_mempoll_id;

/* *******************************************************************
 * RTDS_MPoolCreate()
 * PARAMETERS
 *      size : Size of memory the memory pool
 *-------------------------------------------------------------------
 * The purpose of this function is to initialize a pool of memory
 ******************************************************************** */
void RTDS_MPoolCreate( int size )
    {
    T_CMPL * pk_cmpl = NULL;
    pk_cmpl = ( T_CMPL * )malloc( sizeof( T_CMPL ) );
    
    if( pk_cmpl == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MALLOC )
        }
    
    memset( pk_cmpl, 0, sizeof( T_CMPL ) );
    
    pk_cmpl->mplatr = TA_TFIFO;                     /* The variable-sized memory pool's wait queue will be in FIFO order */ 
    pk_cmpl->mplsz = size;                          /* Variable-sized Memory area size (in bytes)*/
    pk_cmpl->mpl = NULL;                            /* Beginning address variable-sized memory pool area */
    RTDS_mempoll_id = acre_mpl( pk_cmpl );
    RTDS_CheckOSError( RTDS_mempoll_id );
    }

/* *******************************************************************
 * RTDS_Malloc()
 * PARAMETERS
 *      size : Size of memory wanted
 *-------------------------------------------------------------------
 * The purpose of this function is to allocate memory from the memory
 * pool
 ******************************************************************** */
VP RTDS_Malloc( int size )
    {
    VP p_blk;
    ER rvalue;
    
    rvalue = pget_mpl( RTDS_mempoll_id, size, &p_blk ); /*Parameter: (ID mplid, UINT blksz,VP* P_BLK */
    
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PGETMPL )
        }
    
    memset( p_blk, 0, size );
    
    return p_blk;
    }

/* *******************************************************************
 * RTDS_Free()
 * PARAMETERS
 *      poolId : Mempool ID
 *      p_blk : Pointer to previous allocated block memory
 *
 * RETURN
 *      Pointer on the created block memory
 *-------------------------------------------------------------------
 * The purpose of this function is to free allocated memory
 ******************************************************************** */
void RTDS_Free( ID poolId, VP p_blk )
    {
    ER rvalue;
  
    rvalue = rel_mpl( poolId, p_blk );
    
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_RELMPL )
        }
    }

/* *******************************************************************
 * RTDS_SystemInit()
 *
 *-------------------------------------------------------------------
 * The purpose of this function is initialize system
 ******************************************************************** */
void RTDS_SystemInit( void )
    {
    RTDS_MPoolCreate( RTDS_MEMPOLL_SIZE );
    }

/* *******************************************************************
 * RTDS_Sem_Flush (UITRON semaphore function)
 *
 *-------------------------------------------------------------------
 * The purpose of this function is to flush semaphore,
 * allowing threads or process waiting for the semaphore
 * (function parameter) to run
 *
 * function used in file RTDS_OS_basic.h for synchro
 * start semaphore
 *
 ******************************************************************** */
int RTDS_SemFlush( RTDS_SemType * SemId_To_Flush )
    {
    ER rvalue;
    
    rvalue = sig_sem( ( ID )SemId_To_Flush->SemID );
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_FLUSH )
        }
    
    return rvalue;
    }

/* *******************************************************************
 * RTDS_SemBCreate
 *
 *-------------------------------------------------------------------
 * The purpose of this function is to create binary semaphore
*
 * Parameters:
 *      opt : 0:FIFO(TA_TFIFO) or 1:PRIO(TA_TPRIO)
 *      initialState : initial state when creating 0: Empty / 1: Full
 * Returns:
 *      pointer on the created semaphore: SemaphoreId
 ******************************************************************** */
RTDS_SemType * RTDS_SemBCreate( int opt, int initialState )
    {
    RTDS_SemType * SemId = NULL;
    
    SemId = RTDS_SemCCreate( opt, initialState );
    SemId->RTDS_Info.semType = BINARY;
    
    return SemId;
    }

/* *******************************************************************
 * RTDS_SemCCreate
 *
 *-------------------------------------------------------------------
 * The purpose of this function is to create counting semaphore
 *
 * Parameters:
 *      opt : 0:FIFO(TA_TFIFO) or 1:PRIO(TA_TPRIO)
 *      initialState : initial state when creating 0: Empty / 1: Full
 * Returns:
 *      STATUS CODES: E_OK - Normal Completion
 ******************************************************************** */
RTDS_SemType * RTDS_SemCCreate( int opt, int initialState )
    {
    RTDS_SemType * SemId = NULL;
    
    SemId = ( RTDS_SemType * )RTDS_MALLOC( sizeof( RTDS_SemType ) );
    
    /* sematr - Semaphore Attribute Values */
    /* #define TA_TFIFO 0x00 => waiting tasks are handled by FIFO */
    /* #define TA_TPRI 0x01 => waiting tasks are handled by priority */
    SemId->RTDS_Info.confStruct.sematr = opt;           /* ATR semaphore attributes */
    SemId->RTDS_Info.confStruct.isemcnt = initialState; /* INT: initial semaphore count */
    SemId->RTDS_Info.confStruct.maxsem = 100;           /* INT : maximum semaphore count : NOT SUPPORTED */
    
    SemId->RTDS_Info.semType = COUNTING;                /* RTDS Specific*/
    
    /* Create the semaphore with a automatic ID assigment */
    SemId->SemID = ( ID )acre_sem( &SemId->RTDS_Info.confStruct );
    /* Check if any error occur */
    RTDS_CheckOSError( SemId->SemID );
    /* Then return the ID of the created semaphore */
    return SemId;
    }

/* *******************************************************************
 * RTDS_SemDelete
 *
 *-------------------------------------------------------------------
 * The purpose of this function is to create binary semaphore
 *
 * Parameters:
 *      semId : semaphore Id
 *
 * Returns:
 *      function status -1: UITRON_ERROR or 0: UITRON_OK
 ******************************************************************** */
void RTDS_SemDelete( RTDS_SemType * semId )
    {
    ER rvalue;
    
    rvalue = del_sem( ( ID )semId->SemID );
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_DEL )
        }
    }

/* *******************************************************************
 * RTDS_SemGive
 *
 *-------------------------------------------------------------------
 * The purpose of this function is to release semaphore
 *
 * Parameters:
 *      semId : semaphore Id
 *
 * Returns:
 *      function status -1: UITRON_ERROR or 0: UITRON_OK
 ******************************************************************** */
int RTDS_SemGive( RTDS_SemType * semId )
    {
    ER rvalue;
    T_RSEM pk_rsem;
    
    if( semId->RTDS_Info.semType != BINARY )
        {
        rvalue = sig_sem( ( ID )semId->SemID );
        if( rvalue != UITRON_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_SIG )
            }
        return ( int )rvalue;
        }
    
    else
        {
        rvalue = ref_sem( ( ID )semId->SemID, &pk_rsem );
        if ( rvalue != UITRON_OK )
            {
            return UITRON_ERROR;
            }
        if ( pk_rsem.semcnt > 0 )
            {
            return UITRON_OK;
            }
        return sig_sem( ( ID )semId->SemID );
        }
    }

/* ************************************************************************
 * RTDS_SemTake
 *
 *-------------------------------------------------------------------------
 * The purpose of this function is to take semaphore
 *
 * Parameters:
 *      SemId : semaphore Id
 *      time_to_Wait: if 0 no wait otherwise wait until semaphore available
 *
 * Returns:
 *      function status RTDS_ERROR or RTDS_OK
 ******************************************************************** */
int RTDS_SemTake( RTDS_SemType * SemId, long time_to_Wait )
    {
    ER rvalue;
    
    if( time_to_Wait == RTDS_UITRON_SEM_TIMEOUT_FOREVER )
        {
        rvalue = wai_sem( ( ID )SemId->SemID );
        }
    else
        {
        rvalue = twai_sem( ( ID )SemId->SemID,time_to_Wait );
        if( rvalue == E_TMOUT )
            {
            return RTDS_ERROR;
            }
        }
    if( rvalue == UITRON_OK )
        {
        return RTDS_OK;
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_WAI );
        return RTDS_ERROR;
        }
    }

/* *******************************************************************
 * RTDS_TickGet()
 *
 *-------------------------------------------------------------------
 * The purpose of this function is to get the system
 * time (global variable RTDS_SystemTime)
 ******************************************************************* */
long RTDS_TickGet( void )
    {
    ER rvalue;
    SYSTIM sysTime;
    
    rvalue = get_tim( &sysTime );
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_TIM )
        }
    
    return sysTime;
    }

/* *******************************************************************
 * RTDS_TickSet()
 *
 *-------------------------------------------------------------------
 * The purpose of this function is set the system time
 * therefore this function set the extern variable (RTDS_globalSystemTime)
 * defined in the main function using System Time MACRO (RTDS_MACRO.h)
 *
 ******************************************************************* */
void RTDS_TickSet( long timeValue )
    {
    ER rvalue;
    
    SYSTIM newtime = timeValue;
    rvalue = set_tim( &newtime );
    
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SET_TIM )
        }
    }

/* **************************************************************** *
 * RTDS_GetCurrentThread
 * **************************************************************** *
 * Returns the current thread
 * **************************************************************** *
 * Parameters:
 *      ressourceType :
 * Returns:
 *      status operation
 * **************************************************************** */
ID RTDS_GetCurrentThread()
    {
    ID threadId;
    ER rvalue;
    
    rvalue = get_tid( &threadId );
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_TID )
        }
        
    return threadId;
    }

/* **************************************************************** *
 * RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *      timerList: points to the first element of
 * the chained list of RTDS_TimerState
 * Returns:
 *      the unique timer Id
 * **************************************************************** *
 * Context:
 *      The chained list of RTDS_TimerState is sorted by increasing
 *      unique timer id
 *      timerUniqueId = NULL means the message is not a timer; so it
 *      is avoided.
 * **************************************************************** */
long RTDS_GetTimerUniqueId( RTDS_TimerState * timerList )
    {
    RTDS_TimerState * RTDS_prevTimer;
    RTDS_TimerState * RTDS_timer;
    long newTimerId;
    
    RTDS_CRITICAL_SECTION_START;
    /* If list is empty take 1 */
    if( timerList == NULL )
        {
        newTimerId = 1;
        }
    /* If 1 is available, take it */
    else if( timerList->timerUniqueId != 1 )
        {
        newTimerId = 1;
        }
    else
        {
        /* If there is a gap in the timerId chained list, take an Id in the gap */
        RTDS_prevTimer = timerList;
        newTimerId = 0;
        for ( RTDS_timer = timerList->next; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
            {
            if ( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 )
                {
                newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                }
            RTDS_prevTimer = RTDS_timer;
            }
        /* No gap, let's take the next value */
        if ( newTimerId == 0 ) 
            {
            newTimerId = RTDS_prevTimer->timerUniqueId + 1;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Check the counter did not go back to 0 */
    if ( newTimerId == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID );
        }
    
    return newTimerId;
    }

/* **************************************************************** *
 * RTDS_WatchDogFunction
 * **************************************************************** *
 * Send a timer message using the parameters given to the watchdog
 * **************************************************************** *
 * Parameters:
 *      ptimerState: Needed parameters to create and send the timer
 * Returns:
 *      nothing
 * **************************************************************** *
 * Context:
 *      Each time a timer is set, a watchdog is created; that's why it has to destroy itself.
 * **************************************************************** */
void RTDS_WatchDogFunction( RTDS_TimerState * ptimerState )
    {
    ER rvalue;
    T_MSG_PACKET * pBuf = NULL;
    
    /* Send the message */
    pBuf = ( T_MSG_PACKET * )ptimerState->timerMessage;
    
    rvalue = isnd_mbx( ptimerState->receiverId->queueId, ( T_MSG * )pBuf );
    
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG )
        }
    }

/* **************************************************************** *
 * RTDS_forceTimer
 * **************************************************************** *
 * call RTDS_WatchDogFunction(RTDS_TimerState * parameter)
 * Function called by TCP client to force timer to go off.
 * **************************************************************** *
 * Parameters:
 *      processId
 *      timerNumber
 * Returns:
 *      nothing
 * **************************************************************** */
#ifdef RTDS_SIMULATOR

void RTDS_forceTimer( RTDS_RtosTaskId processId, int timerNumber )
    {
    RTDS_GlobalProcessInfo * processInfo;
    RTDS_TimerState * timerInfo = NULL;
    RTDS_TimerState * timerState = NULL;
    long timeOut = 0;
    int status = 0;
    
    RTDS_CRITICAL_SECTION_START;
    /* Find timerState (RTDS timer ID) from its timer number and the related processId */
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->myRtosTaskId == processId )
            {
            for ( timerInfo = processInfo->timerList; timerInfo != NULL; timerInfo = timerInfo->next )
                {
                if ( timerInfo->timerNumber == timerNumber && timerInfo->state == RTDS_TIMER_OK )
                    {
                    timerState = timerInfo;
                    timeOut = timerState->timeoutValue;
                    break;
                    }
                }
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Call timer function to send SDL message */
    if( timerState != NULL )
        {
        /* Let's try to cancel the watchdog */
        status = del_alm( timerState->watchDogId );
        if( status != UITRON_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_DEL_ALM )
            }
        
        RTDS_WatchDogFunction( timerState );
#ifndef RTDS_RT_TIMER
        RTDS_TickSet( timeOut );
#endif /* RTDS_RT_TIMER */
        }
    }
#endif /* RTDS_SIMULATOR */

/* **************************************************************** *
 * RTDS_StartTimer
 * **************************************************************** *
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *      queueId of the receiver
 *      TimerNumber
 *      timerUniqueId
 *       delay of the timer
 *       timerStateList: Address of the list of timer
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_StartTimer( RTDS_SdlInstanceId * instance, long timerNumber, long timerUniqueId,int delay, RTDS_TimerState ** pTimerStateList, RTDS_GlobalProcessInfo * currentContext)
    {
    RTDS_TimerState * timerState;
    RTDS_TimerState * previousTimerState;
    RTDS_TimerState * newTimerState;
    T_CALM * configTimer = NULL;
    RTDS_TimerId watchDogId;
    long timeSetting;
    
    /* Update the list of timers */
    /* Important note: it is a sorted list based on the timerUniqueId field */
    newTimerState = ( RTDS_TimerState * )RTDS_MALLOC( sizeof( RTDS_TimerState ) );
    if( newTimerState == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MALLOC_TIMER_STATE_IN_START_TIMER );
        }
        
    /* Initialize the new element */
    newTimerState->state = RTDS_TIMER_OK;
    newTimerState->timerNumber = timerNumber;
    newTimerState->timeoutValue = delay + RTDS_TickGet() - RTDS_globalStartTime;
    newTimerState->timerUniqueId = timerUniqueId;
    newTimerState->next = NULL;
    newTimerState->receiverId = instance;
    newTimerState->timerMessage = ( T_MSG_PACKET * )RTDS_MALLOC( sizeof( T_MSG_PACKET ) );
    newTimerState->timerMessage->msgHeader = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );
    
    /* Fill in the message parameters */
#ifdef RTDS_SIMULATOR
    newTimerState->timerMessage->msgHeader->messageUniqueId = 0;
#endif
    
    newTimerState->timerMessage->msgHeader->messageNumber = newTimerState->timerNumber;
    newTimerState->timerMessage->msgHeader->timerUniqueId = newTimerState->timerUniqueId;
    newTimerState->timerMessage->msgHeader->sender = newTimerState->receiverId;
    newTimerState->timerMessage->msgHeader->receiver = newTimerState->receiverId;
    newTimerState->timerMessage->msgHeader->next = NULL;
    newTimerState->timerMessage->msgHeader->dataLength = 0;
    newTimerState->timerMessage->msgHeader->pData = NULL;
    newTimerState->timerMessage->msgHeader->next = NULL;
    
    /* Insert the new element */
    if ( *pTimerStateList == NULL ) /* The list is empty */
        {
        *pTimerStateList = newTimerState;
        }
    else /* The list is not empty */
        {
        previousTimerState = NULL;
        for ( timerState = *pTimerStateList; timerState != NULL; timerState = timerState->next )
            {
            if ( timerState->timerUniqueId > timerUniqueId )
                {
                if (previousTimerState == NULL)
                    {
                    *pTimerStateList = newTimerState;
                    newTimerState->next = timerState;
                    }
                else
                    {
                    previousTimerState->next = newTimerState;
                    newTimerState->next = timerState;
                    }
                break;
                }
            previousTimerState = timerState;
            }
        
        if ( timerState == NULL ) /* Inserted at the end of the list */
            {
            previousTimerState->next = newTimerState;
            }
        }
    
    RTDS_SIMULATOR_TRACE( RTDS_timerStarted, newTimerState, delay, currentContext );
    
    configTimer= ( T_CALM * )RTDS_MALLOC( sizeof( T_CALM ) );
    configTimer->exinf = ( VP_INT )newTimerState;       /* alarm handler parameter */
    configTimer->almatr = ( ATR )TA_HLNG;               /* alarm handler attributes */
    configTimer->almhdr = ( FP )RTDS_WatchDogFunction;  /* alarm handler address */
    
    newTimerState->alarmControl = configTimer;
    
    /* Create the timer */
    watchDogId = acre_alm( configTimer );
    
    if( watchDogId < 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_CREATION );
        }
    else
        {
        newTimerState->watchDogId = watchDogId;
        /* Start alarm */
        timeSetting = delay;
        if ( timeSetting < 1 )
            {
            timeSetting = 1;
            }
#ifdef RTDS_FAKE_TIMER
        timeSetting = 1;
#endif
        if ( sta_alm( watchDogId, timeSetting ) < 0 )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_START );
            }
        }
    }

/* **************************************************************** *
 * RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *      timerNumber
 *      pTimerStateList pointing to the timer chained list
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_StopTimer( long timerNumber, RTDS_TimerState ** pTimerStateList, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState * RTDS_timer;
    RTDS_TimerState * RTDS_prevTimer;
    RTDS_MessageHeader * RTDS_message;
    RTDS_MessageHeader * RTDS_prevMessage;
    
    RTDS_prevTimer = NULL;
    for( RTDS_timer = *pTimerStateList; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
        {
        if( ( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ) )
            {
            RTDS_SIMULATOR_TRACE( RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext );
            
            /* Let's try to cancel the watchdog */
            if ( del_alm( RTDS_timer->watchDogId ) == UITRON_OK )
                {
                RTDS_FREE( RTDS_timer->timerMessage->msgHeader );
                RTDS_FREE( RTDS_timer->timerMessage );
                RTDS_FREE( RTDS_timer->alarmControl );
                /* Remove it from the list */
                /* Is the first of the list */
                if ( RTDS_prevTimer == NULL )
                    {
                    *pTimerStateList = RTDS_timer->next;
                    }
                else /* Not the first of the list */
                    {
                    RTDS_prevTimer->next = RTDS_timer->next;
                    }
                RTDS_FREE( RTDS_timer );
                }
            /* Could not cancel the timer. Probably went off allready */
            /* Set it cancelled in the list */
            else
                {
                RTDS_timer->state = RTDS_TIMER_CANCELLED;
                }
            return;
            }
        RTDS_prevTimer = RTDS_timer;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->readSaveQueue ; RTDS_message!=NULL ; RTDS_message=RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->readSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            RTDS_FREE( RTDS_message );
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for ( RTDS_message = RTDS_currentContext->writeSaveQueue; RTDS_message!=NULL; RTDS_message=RTDS_message->next )
        {
        if ( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if ( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->writeSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            RTDS_FREE( RTDS_message );
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    }

/* **************************************************************** *
 * RTDS_GetProcessInstanceId
 * **************************************************************** *
 * Returns the queueId of process based on its name
 * **************************************************************** *
 * Parameters:
 *      process name as a number defined in RTDS_gen.h
 * Returns:
 *      the process queue id if found
 * Error:
 *      system error call if not found and NULL returned
 * **************************************************************** */
RTDS_SdlInstanceId * RTDS_GetProcessInstanceId( int processNumber )
    {
    RTDS_GlobalProcessInfo * processInfo;
    RTDS_SdlInstanceId * foundQueue = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->sdlProcessNumber == processNumber )
            {
            foundQueue = processInfo->mySdlInstanceId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( foundQueue == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_PROCESS_QUEUE_ID );
        }
    
    return foundQueue;
    }

/* **************************************************************** *
 * RTDS_CheckInstanceId
 * **************************************************************** *
 * Returns UITRON_OK is queueId exist or ERROR if not
 * **************************************************************** *
 * Parameters:
 *      Queue Id
 * Returns:
 *      UITRON_OK/ERROR
 * **************************************************************** */
int RTDS_CheckInstanceId( RTDS_SdlInstanceId * instance )
    {
    RTDS_GlobalProcessInfo * processInfo;
    
    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if ( processInfo->mySdlInstanceId == instance )
            {
            RTDS_CRITICAL_SECTION_STOP;
            return UITRON_OK;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    return UITRON_ERROR;
}

/* **************************************************************** *
 * RTDS_MsgSend
 * **************************************************************** *
 * Send a message in a process's queue
 * **************************************************************** *
 * Parameters:
 *      messageNumber representing a message name
 *      dataLength length of data pointed by pData
 *      pData pointer on data sent with the message
 *      message receiver queue address
 *      message sender queue address
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_MsgSend
(
long messageNumber,
long dataLength,
unsigned char * pData,
RTDS_SdlInstanceId * receiver,
RTDS_SdlInstanceId * sender,
RTDS_GlobalProcessInfo * currentContext
)
    {
    ER rvalue = 0;
    RTDS_MessageHeader * RTDS_ptrMessageToSend;
    T_MSG_PACKET * pBuf = NULL;
    
    RTDS_ptrMessageToSend = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );
    
    RTDS_ptrMessageToSend->messageNumber = messageNumber;
    RTDS_ptrMessageToSend->timerUniqueId = 0;
    RTDS_ptrMessageToSend->sender = sender;
    RTDS_ptrMessageToSend->receiver = receiver;
    RTDS_ptrMessageToSend->dataLength = dataLength;
    RTDS_ptrMessageToSend->pData = pData;
    RTDS_ptrMessageToSend->next = NULL;
    
#ifdef RTDS_SIMULATOR
    RTDS_ptrMessageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
#endif
    
    RTDS_SIMULATOR_TRACE( RTDS_messageSent, RTDS_ptrMessageToSend, receiver, currentContext );
    
    pBuf = ( T_MSG_PACKET * )RTDS_MALLOC( sizeof( T_MSG_PACKET ) );
    pBuf->msgHeader = RTDS_ptrMessageToSend;
    
    rvalue = snd_mbx( receiver->queueId, ( T_MSG * )pBuf );
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }

/* **************************************************************** *
 * RTDS_SimulatorMsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue from the simulator
 * Basically the same as RTDS_MsgQueueSend but it gets the
 * MsgUniqueId as a parameter and does not call the dummy trace
 * function
 * **************************************************************** *
 * Parameters:
 *      messageNumber representing a message name
 *      dataLength length of data pointed by pData
 *      pData pointer on data sent with the message
 *      receiver message receiver queue address
 *      sender message sender queue address
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_SimulatorMsgQueueSend
(
long messageNumber,
long dataLength,
unsigned char * pData,
RTDS_SdlInstanceId * receiver,
RTDS_SdlInstanceId * sender
)
    {
    ER rvalue;
    RTDS_MessageHeader * RTDS_ptrMessageToSend;
    T_MSG_PACKET * pBuf = NULL;
    
    RTDS_ptrMessageToSend = ( RTDS_MessageHeader * )RTDS_MALLOC( sizeof( RTDS_MessageHeader ) );
    
    RTDS_ptrMessageToSend->messageNumber = messageNumber;
    RTDS_ptrMessageToSend->timerUniqueId = 0;
    RTDS_ptrMessageToSend->sender = sender;
    RTDS_ptrMessageToSend->receiver = receiver;
    RTDS_ptrMessageToSend->dataLength = dataLength;
    RTDS_ptrMessageToSend->pData = pData;
    RTDS_ptrMessageToSend->next = NULL;
#ifdef RTDS_SIMULATOR
    RTDS_ptrMessageToSend->messageUniqueId = RTDS_GetMessageUniqueId();
#endif
    
    /* Specific for the backTrace since the original trace comes from the SDL-RT debugger */
#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
    RTDS_globalTraceEntry.event = RTDS_messageSent;
    RTDS_globalTraceEntry.eventParameter1 = ( void * )RTDS_ptrMessageToSend;
    RTDS_globalTraceEntry.eventParameter2 = ( long )receiver;
    RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )NULL;
    RTDS_TraceAdd();
#endif
    
    pBuf = ( T_MSG_PACKET * )RTDS_MALLOC( sizeof( T_MSG_PACKET ) );
    pBuf->msgHeader = RTDS_ptrMessageToSend;
    rvalue = snd_mbx( receiver->queueId, ( T_MSG * )pBuf );
    
    if( rvalue != UITRON_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }

/* **************************************************************** *
 * RTDS_MsgReceive
 * **************************************************************** *
 * Receives a message from a process instance
 * **************************************************************** *
 * Parameters:
 *        instance id to read the messsage from
 *        pointer on the message to receive
 *        size of message
 *        mode used to read (blocking or non blocking)
 * Returns:
 *       Nothing
 * **************************************************************** */
void RTDS_MsgReceive( RTDS_SdlInstanceId * instanceId, RTDS_MessageHeader ** message )
    {
    /* Receive the message */
    if( RTDS_MsgQueueReceive( instanceId->queueId, message ) != RTDS_OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_RECEIVE )
        }
    }

/* **************************************************************** *
 * RTDS_MsgQueueReceive
 * **************************************************************** *
 * Receives a message from a process's queue
 * **************************************************************** *
 * Parameters:
 *      queue id to read the messsage from
 *      pointer on a pointer on the message to receive
 * Returns:
 *      nothing
 * **************************************************************** */
int RTDS_MsgQueueReceive( RTDS_RtosQueueId queueId, RTDS_MessageHeader ** message )
    {
    ER rvalue = 0;
    T_MSG_PACKET * msgPtr;
    T_MSG * pData = NULL;
    
    rvalue = rcv_mbx( queueId, &pData );
    if ( rvalue == UITRON_ERROR )
        {
        return RTDS_ERROR;
        }
    
    msgPtr = ( T_MSG_PACKET * )pData;
    *message = msgPtr->msgHeader;
    RTDS_FREE( pData );
    
    return RTDS_OK;
    }

/* *******************************************************************
 * RTDS_TaskCreate
 *-------------------------------------------------------------------
 * Creates a new task
 *
 * Parameter:
 *      priority : Message queue to delete
 *      functionAddress : Adresse of function to execute
 *      newProcessInfo : Parameters for the created thread
 * Returns:
 *      RTDS_RtosTaskId : ID of the created task 
 ******************************************************************** */
RTDS_RtosTaskId RTDS_TaskCreate( int priority,  void * functionAddress, void * newProcessInfo )
    {
    RTDS_RtosTaskId processId;
    T_CTSK * confStruct = NULL;
    
    confStruct = ( T_CTSK * )RTDS_MALLOC( sizeof( T_CTSK ) );
    confStruct->tskatr = TA_HLNG | TA_ACT;
    confStruct->exinf = ( VP_INT )newProcessInfo;
    confStruct->task = ( FP )functionAddress;
    confStruct->itskpri = priority; 
    confStruct->stksz =  RTDS_TASK_STACK_SIZE; 
    confStruct->stk = RTDS_MALLOC( RTDS_TASK_STACK_SIZE );
    
    processId = acre_tsk( confStruct );
    RTDS_CheckOSError( processId );
    
    return processId; 
    }

/* **************************************************************** *
 * RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *      name of the process as a number
 *      address of the process function
 *      priority fo the process
 *      the address of the address of the msg queue to be created
 *      the address of the msg queue of the parent process
 * Returns:
 *      nothing but updates the OFFSPRING of the caller
 * **************************************************************** */
void RTDS_ProcessCreate( int processNumber, void * functionAddress, int priority, short synchronization, RTDS_GlobalProcessInfo * currentContext )
    {
    T_CTSK * confStruct = NULL;
    T_CMBX * pk_cmbx = NULL;
    RTDS_GlobalProcessInfo * processInfo = NULL;
    RTDS_GlobalProcessInfo * newProcessInfo = NULL;
    
    /* Allocate and fill in a new processInfo structure */
    newProcessInfo = ( RTDS_GlobalProcessInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalProcessInfo ) );
    if ( newProcessInfo == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_MALLOC );
        }
    
    /* Sets the process name as a number */
    newProcessInfo->sdlProcessNumber = processNumber;
    /* SDL initial state */
    newProcessInfo->sdlState = 0;
    /* Set SDL task priority */
    newProcessInfo->priority = priority;
    
    /* Last structure of the chained list */
    newProcessInfo->next = NULL;
    newProcessInfo->next = NULL;
    newProcessInfo->parentSdlInstanceId = currentContext->mySdlInstanceId;
    newProcessInfo->offspringSdlInstanceId = 0;
    newProcessInfo->currentMessage = NULL;
    newProcessInfo->timerList = NULL;

    newProcessInfo->mySdlInstanceId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );
    
    /* New task's queue creation */
    pk_cmbx = ( T_CMBX * )RTDS_MALLOC( sizeof( T_CMBX ) );
    pk_cmbx->mbxatr = TA_MFIFO;
    pk_cmbx->mprihd = NULL;
    newProcessInfo->mySdlInstanceId->queueId = acre_mbx( pk_cmbx );
    RTDS_CheckOSError( newProcessInfo->mySdlInstanceId->queueId );
    
    currentContext->offspringSdlInstanceId = newProcessInfo->mySdlInstanceId;
    newProcessInfo->taskQueue = pk_cmbx;
    
    /* Take the synchronization semaphore so that the created task does not run straigh away */
    if ( synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_HOLD;
        }
    /* Task creation */
    confStruct = ( T_CTSK * )RTDS_MALLOC( sizeof( T_CTSK ) );
    confStruct->tskatr = TA_HLNG | TA_ACT;
    confStruct->exinf = ( VP_INT )newProcessInfo;
    confStruct->task = ( FP )functionAddress;
    confStruct->itskpri = priority; 
    confStruct->stksz = RTDS_TASK_STACK_SIZE; 
    confStruct->stk = RTDS_MALLOC( RTDS_TASK_STACK_SIZE );
    
    newProcessInfo->myRtosTaskId = acre_tsk( confStruct );
    RTDS_CheckOSError( newProcessInfo->myRtosTaskId );
    newProcessInfo->taskStruct = confStruct;
    
    /* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalProcessInfo == NULL )
        {
        RTDS_globalProcessInfo = newProcessInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for ( processInfo = RTDS_globalProcessInfo; processInfo->next != NULL; processInfo = processInfo->next );
        processInfo->next = newProcessInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_processCreated, newProcessInfo, NULL, currentContext );
    /* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
    if ( synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_UNHOLD;
        }
    }

/* **************************************************************** *
 * RTDS_MsgQueueCreate
 * **************************************************************** *
 * Create a message queue
 * **************************************************************** *
 * Parameters:
 *        nothing
 * Returns:
 *       the created message queue ID
 * **************************************************************** */
RTDS_RtosQueueId RTDS_MsgQueueCreate( void )
    {
    T_CMBX * pk_cmbx = NULL;
    RTDS_RtosQueueId mbxId;
    
    pk_cmbx = ( T_CMBX * )RTDS_MALLOC( sizeof( T_CMBX ) );
    pk_cmbx->mbxatr = TA_MFIFO;
    pk_cmbx->mprihd = NULL;
    mbxId = acre_mbx( pk_cmbx );
    RTDS_CheckOSError( mbxId );
    
    return mbxId;
    }

/* **************************************************************** *
 * RTDS_ProcessForget
 *-------------------------------------------------------------------
 * Forgets a given process, freeing all associated information
 *
 * Parameters:
 *  - RTDS_currentContext : The current context of the task to delete
 *  - deleteQueue: If true, the message queue for the process is destroyed
 * Returns:
 *  - The PID for the forgotten process or NULL if process was not found
 * **************************************************************** */
RTDS_RtosTaskId RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext, short deleteQueue )
    {
    RTDS_GlobalProcessInfo * processInfo;
    RTDS_GlobalProcessInfo * previousProcessInfo;
    RTDS_TimerState * RTDS_timer;
    RTDS_MessageHeader * message;
    RTDS_RtosQueueId queuetoDelete;
    int pidToKill = 0;
    ER rvalue;
    
    /* Let's free the current message if any */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        #ifdef RTDS_SIMULATOR
          /* Release the message unique id back to the pool */
          RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        }
    
    /* Let's clean the timer chained list to free memory and stop watchdogs */
    for ( RTDS_timer = RTDS_currentContext->timerList; RTDS_timer != NULL; RTDS_timer = RTDS_currentContext->timerList )
        {
        rvalue = del_alm( RTDS_timer->watchDogId );
        if( rvalue!=UITRON_OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_DEL_ALM )
            }
        
        RTDS_currentContext->timerList = RTDS_timer->next;
        RTDS_FREE( RTDS_timer );
        }
    
    /* Clean the save queue: free messages and message unique ids in the read and write save queues */
    for ( message = RTDS_currentContext->readSaveQueue; message != NULL; message = message->next )
        {
#ifdef RTDS_SIMULATOR
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
#endif
        RTDS_FREE( message );
        }
    
    for ( message = RTDS_currentContext->writeSaveQueue; message != NULL; message = message->next )
        {
#ifdef RTDS_SIMULATOR
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
#endif
        RTDS_FREE( message );
        }
    previousProcessInfo = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        /* The queue id is used to find the process id */
        if ( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )    /* Found ! */
            {
            RTDS_SIMULATOR_TRACE( RTDS_processDied, processInfo, NULL, RTDS_currentContext );
            /* Update the process information chained list */
            if ( previousProcessInfo == NULL )  /* First one in the list */
                {
                RTDS_globalProcessInfo = processInfo->next;
                }
            else
                {
                previousProcessInfo->next = processInfo->next;
                }
            pidToKill = processInfo->myRtosTaskId;
            /* Delete the message queue and free the process info block */
             if ( deleteQueue )
                {
                queuetoDelete = processInfo->mySdlInstanceId->queueId;
                if ( del_mbx( queuetoDelete ) != UITRON_OK )
                    {
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_DELETE );
                    }
                RTDS_FREE( processInfo->taskStruct->stk );
                RTDS_FREE( processInfo->taskStruct );
                RTDS_FREE( processInfo->taskQueue );
                }
            RTDS_FREE( processInfo );
            break;
            }
        previousProcessInfo = processInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    return pidToKill;
    }

/* **************************************************************** *
 * RTDS_ProcessKill
 * **************************************************************** *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *      the address of the msg queue to find the process info block
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_ProcessKill( RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_RtosTaskId pidToKill = NULL;
    
    /* Forget everything about process, delete its message queue and get its PID */
    pidToKill = RTDS_ProcessForget( RTDS_currentContext, 1 );
    
    /* Delete the task */
    if ( pidToKill != 0 )
        {
        ext_tsk();
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND );
        }
    }

/* **************************************************************** *
 * RTDS_Sem_Info_Insert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *      name of the semaphore
 *      id of the semaphore
 * Returns:
 *      nothing
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Sem_Info_Insert( int semaphoreNumber, RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * currentContext)
    {
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_GlobalSemaphoreInfo * newSemInfo;
    
    /* Allocate and fill in a new semInfo structure */
    newSemInfo = ( RTDS_GlobalSemaphoreInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalSemaphoreInfo ) );
    if ( newSemInfo == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT_MALLOC );
        }
    
    newSemInfo->semaphoreNumber = semaphoreNumber;
    newSemInfo->next = NULL;
    
    /* Semaphore creation */
    newSemInfo->semaphoreId = semaphoreId;
    if ( newSemInfo->semaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT );
        }
    
    /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if ( RTDS_globalSemaphoreInfo == NULL )
        {
        RTDS_globalSemaphoreInfo = newSemInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for ( semInfo = RTDS_globalSemaphoreInfo; semInfo->next != NULL; semInfo = semInfo->next );
        semInfo->next = newSemInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreCreated, semaphoreId, 0, currentContext );

    return semaphoreId;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdDelete
 * **************************************************************** *
 * Kills a semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *      the address of the semaphore to find the semaphore info block
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_SemaphoreIdDelete( RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_GlobalSemaphoreInfo * semInfoPrev;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreDeleted, semaphoreId, 0, currentContext );
    
    /* Remove the semaphore information from the chained list */
    semInfoPrev = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        if( semInfo->semaphoreId == semaphoreId ) /* semaphore found */
            {
            if(semInfoPrev == NULL)
                {
                RTDS_globalSemaphoreInfo = semInfo->next;
                }
            else
                {
                semInfoPrev->next = semInfo->next;
                }
            RTDS_SemDelete( semInfo->semaphoreId );
            RTDS_FREE( semInfo );
            break;
            }
        semInfoPrev = semInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *      the id of the semaphore
 *      timeout value NOT USED HERE
 * Returns:
 *      status
 * **************************************************************** */
RTDS_SemaphoreStatus RTDS_SemaphoreIdTake( RTDS_SemaphoreId semaphoreId, RTDS_SemaphoreTimeout TIME_OUT, RTDS_GlobalProcessInfo * currentContext )
    {
    RTDS_SemaphoreStatus tmpStatus;
    
    RTDS_SIMULATOR_TRACE( RTDS_semTakeAttempt, semaphoreId, TIME_OUT, currentContext );
    tmpStatus = RTDS_SemTake( semaphoreId, TIME_OUT );
    if ( tmpStatus == RTDS_OK )
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeSucceded, ( void * )semaphoreId, NULL, currentContext );
        }
    else
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeTimedOut, semaphoreId, NULL, currentContext );
        }
    return tmpStatus;
    }

/* **************************************************************** *
 * RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *      semaphore number representing its name
 * Returns:
 *      the id of the semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
    {
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_SemaphoreId foundSemaphoreId;
    
    foundSemaphoreId = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for ( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        if ( semInfo->semaphoreNumber == semaphoreNumber )  /* semaphore found */
            {
            foundSemaphoreId = semInfo->semaphoreId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if ( foundSemaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_SEMAPHORE_ID );
        }
        
    return foundSemaphoreId;    /* Might cause an RTOS exception if NULL*/
    }

/* **************************************************************** *
 * RTDS_CheckOSError
 * **************************************************************** *
 * Check if uItron error is detected
 * **************************************************************** *
 * Parameters:
 *      error code
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_CheckOSError( int code )
    {
    switch ( code )
        {
        case E_ID : RTDS_SYSTEM_ERROR( RTDS_E_ID );
        case E_NOID : RTDS_SYSTEM_ERROR( RTDS_E_NOID );
        case E_NOMEM : RTDS_SYSTEM_ERROR( RTDS_E_NOMEM );
        case E_RSATR: RTDS_SYSTEM_ERROR( RTDS_E_RSATR );
        case E_PAR : RTDS_SYSTEM_ERROR( RTDS_E_PAR );
        case E_OBJ : RTDS_SYSTEM_ERROR( RTDS_E_OBJ );
        }
    }

/* **************************************************************** *
 * RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * - frees message buffer if valid
 * - re-organize save queue if state has changed
 * **************************************************************** *
 * Parameters:
 *      none
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_TransitionCleanUp( RTDS_GlobalProcessInfo * RTDS_currentContext, int RTDS_sdlStatePrev )
    {
    /* Free message buffer if valid */
    if ( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        #ifdef RTDS_SIMULATOR
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        RTDS_currentContext->currentMessage = NULL;
        } /* End of if ( RTDS_currentContext->currentMessage != NULL ) */
    
    /* If SDL state has changed and messages have been saved: reorganise the save queue */
    if ( ( RTDS_currentContext->sdlState != RTDS_sdlStatePrev ) && ( RTDS_currentContext->writeSaveQueue != NULL ) )
        {
        RTDS_MessageHeader * message;
        /* Let's get to the end of the save queue */
        for ( message = RTDS_currentContext->writeSaveQueue; message->next != NULL; message = message->next );
        message->next = RTDS_currentContext->readSaveQueue;
        RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
        RTDS_currentContext->writeSaveQueue = NULL;
        }
    
    return;
    }

/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
    char * RTDS_globalPrintableParameters = NULL;
    void * RTDS_globalMessageUniqueIdPool = NULL;
    RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError, NULL, 0, NULL };
#endif

/* **************************************************************** *
 * RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *      none
 * Returns:
 *      the message unique id (minimum is 1)
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
unsigned long RTDS_GetMessageUniqueId( void )
    {
    unsigned char * index;
    long uniqueByteId;
    long uniqueBitId;
    
    index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;
    
    RTDS_CRITICAL_SECTION_START;
    for ( uniqueByteId = 0; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE; uniqueByteId++ )
        {
        if ( *index != 0xFF )
            {
            for ( uniqueBitId = 0; uniqueBitId < 8; uniqueBitId++ )
                {
                if ( ( (1 << uniqueBitId ) & *index ) == 0 )
                    {
                    *index = *index | ( 1 << uniqueBitId );
                    RTDS_CRITICAL_SECTION_STOP;
                    return ( 8 * uniqueByteId + uniqueBitId + 1 );
                    }
                }
            }
        index++;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* All bits are set... No more message unique id */
    RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID );
    return 0;
    }
#endif /* RTDS_SIMULATOR */

/* **************************************************************** *
 * RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *      message unique id
 * Returns:
 *      nothing
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
    {
    unsigned char * index;
    
    if ( messageUniqueId == 0 )
        {
        return;
        }
    messageUniqueId -= 1;
    index = ( unsigned char * )RTDS_globalMessageUniqueIdPool;
    index += ( unsigned char )( messageUniqueId / 8 );
    
    RTDS_CRITICAL_SECTION_START;
    ( *index ) = ( *index ) ^ ( 1 << messageUniqueId % 8 );
    RTDS_CRITICAL_SECTION_STOP;
    
    }
#endif /* RTDS_SIMULATOR */

/* **************************************************************** *
 * RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *      none
 * Returns:
 *      system tick count value
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
unsigned long RTDS_GetSystemTime( void )
    {
    return RTDS_TickGet();
    }
#endif /* RTDS_SIMULATOR */

/* **************************************************************** *
 * RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *      none
 * Returns:
 *      nothing
 * **************************************************************** */
#ifdef RTDS_SIMULATOR
void RTDS_DummyTraceFunction( void )
    {
#ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
    RTDS_TraceAdd();
#endif
    }

#endif /* RTDS_SIMULATOR */
