/* PragmaDev RTDS VxWorks integration */

#include "RTDS_MACRO.h"

/* **************************************************************** *
 * RTDS_GetTimerUniqueId
 * **************************************************************** *
 * Get an available timer unique id.
 * Is used to set timers
 * **************************************************************** *
 * Parameters:
 *      timerList: points to the first element of the chained list of RTDS_TimerState
 * Returns:
 *      The unique timer Id
 * **************************************************************** *
 * Context:
 *      The chained list of RTDS_TimerState is sorted by increasing
 *      unique timer id
 *      timerUniqueId = NULL means the message is not a timer; so it
 *      is avoided.
 * **************************************************************** */
long RTDS_GetTimerUniqueId( RTDS_TimerState * timerList )
    {
    RTDS_TimerState * RTDS_prevTimer;
    RTDS_TimerState * RTDS_timer;
    long newTimerId;
    
    RTDS_CRITICAL_SECTION_START;
    /* If list is empty take 1 */
    if( timerList == NULL )
        {
        newTimerId = 1;
        }
    /* If 1 is available, take it */
    else if( timerList->timerUniqueId != 1 )
        {
        newTimerId = 1;
        }
    else
        {
        /* If there is a gap in the timerId chained list, take an Id in the gap */
        RTDS_prevTimer = timerList;
        newTimerId = 0;
        for( RTDS_timer = timerList->next; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
            {
            if( RTDS_timer->timerUniqueId != RTDS_prevTimer->timerUniqueId + 1 )
                {
                newTimerId = RTDS_prevTimer->timerUniqueId + 1;
                }
            RTDS_prevTimer = RTDS_timer;
            }
        /* No gap, let's take the next value */
        if( newTimerId == 0 )
            {
            newTimerId = RTDS_prevTimer->timerUniqueId + 1;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* Check the counter did not go back to 0 */
    if( newTimerId == 0 )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_TIMER_UNIQUE_ID );
        }
    
    return newTimerId;
    }

/* **************************************************************** *
 * RTDS_WatchDogFunction
 * **************************************************************** *
 * Send a timer message using the parameters given to the watchdog
 * **************************************************************** *
 * Parameters:
 *      parameter: Needed parameters to create and send the timer:
 *          QueueId of the receiver
 *          TimerNumber
 *          TimerUniqueId
 *          WatchDogId to destroy it
 * Returns:
 *      Nothing
 * **************************************************************** *
 * Context:
 *     Each time a timer is set, a watchdog is created; that's why it
 *     has to destroy itself.
 * **************************************************************** */
void RTDS_WatchDogFunction( RTDS_TimerState * parameter )
    {
    RTDS_MessageHeader newMessage;
    
    /* Fill in the message parameters */
    #ifdef RTDS_SIMULATOR
        newMessage.messageUniqueId = 0;
    #endif
    newMessage.messageNumber = parameter->timerNumber;
    newMessage.timerUniqueId = parameter->timerUniqueId;
    newMessage.receiver = parameter->receiverId;
    newMessage.sender = parameter->receiverId;
    newMessage.dataLength = 0;
    newMessage.pData = NULL;
    newMessage.next = NULL;
    
    /* Send the message */
    if( msgQSend( ( MSG_Q_ID )parameter->receiverId->queueId, ( char * )&newMessage, sizeof( RTDS_MessageHeader ), NO_WAIT, MSG_PRI_NORMAL ) == ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_Q_SEND_IN_WATCHDOG );
        }
        
    /* We have to release all memory hold by the watchdog */
    /* So we set its state to finished */
    parameter->state = RTDS_TIMER_FINISHED;
    }

/* **************************************************************** *
 *    RTDS_StartTimer
 * **************************************************************** *
 * Starts a watchdog with the necessary parameters to create the
 * timer when it goes off.
 * **************************************************************** *
 * Parameters:
 *      QueueId of the receiver
 *      TimerNumber
 *      TimerUniqueId
 *      delay of the timer
 *      timerStateList: Address of the list of timer
 *      currentContext (RTDS_GlobalProcessInfo *): active task context
 * Returns:
 *     Nothing
 * **************************************************************** */
void RTDS_StartTimer
( RTDS_SdlInstanceId * instanceId,
 long timerNumber,
 long timerUniqueId,
 int delay,
 RTDS_TimerState ** pTimerStateList,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    WDOG_ID watchDogId;
    RTDS_TimerState * timerState;
    RTDS_TimerState * previousTimerState;
    RTDS_TimerState * newTimerState;
    
    /* Create a new watchdog */
    watchDogId = wdCreate();
    if( watchDogId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_CREATION );
        }
    
    /* Update the list of timers */
    /* Important note: it is a sorted list based on the timerUniqueId field */
    newTimerState = ( RTDS_TimerState * )RTDS_MALLOC( sizeof( RTDS_TimerState ) );
    if( newTimerState == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MALLOC_TIMER_STATE_IN_START_TIMER );
        }
    
    /* Initialize the new element */
    newTimerState->state = RTDS_TIMER_OK;
    newTimerState->timerNumber = timerNumber;
    newTimerState->timeoutValue = delay + tickGet(); /* Only way I found to follow up the timer */
    newTimerState->timerUniqueId = timerUniqueId;
    newTimerState->next = NULL;
    newTimerState->watchDogId = watchDogId;
    newTimerState->receiverId = instanceId;
    
    /* Insert the new element */
    if( *pTimerStateList == NULL ) /* The list is empty */
        {
        *pTimerStateList = newTimerState;
        }
    else    /* The list is not empty */
        {
        previousTimerState = NULL;
        for( timerState = *pTimerStateList; timerState != NULL; timerState = timerState->next )
            {
            if( timerState->timerUniqueId > timerUniqueId )
                {
                if( previousTimerState == NULL )
                    {
                    *pTimerStateList = newTimerState;
                    newTimerState->next = timerState;
                    }
                else
                    {
                    previousTimerState->next = newTimerState;
                    newTimerState->next = timerState;
                    }
                break;
                }
            previousTimerState = timerState;
            }
        if( timerState == NULL ) /* Inserted at the end of the list */
            {
            previousTimerState->next = newTimerState;
            }
        }
    RTDS_SIMULATOR_TRACE( RTDS_timerStarted, newTimerState, delay, RTDS_currentContext );
    /* Starts the watchdog */
    if( wdStart( watchDogId, delay, ( FUNCPTR )RTDS_WatchDogFunction, ( int )newTimerState ) == ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_START );
        }
    }

/* **************************************************************** *
 * RTDS_StopTimer
 * **************************************************************** *
 * Stops a timer in trying to delete the watchdog. If unsuccessfull
 * set it cancelled in the timer chained list verified by the RTDS
 * kernel
 * **************************************************************** *
 * Parameters:
 *      TimerNumber
 *      pTimerStateList pointing to the timer chained list
 *      RTDS_currentContext currentContext of the task
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_StopTimer( long timerNumber, RTDS_TimerState ** pTimerStateList, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_TimerState * RTDS_timer;
    RTDS_TimerState * RTDS_prevTimer;
    RTDS_MessageHeader * RTDS_message;
    RTDS_MessageHeader * RTDS_prevMessage;
    
    RTDS_prevTimer = NULL;
    for( RTDS_timer = *pTimerStateList; RTDS_timer != NULL; RTDS_timer = RTDS_timer->next )
        {
        if( ( RTDS_timer->timerNumber == timerNumber ) && ( RTDS_timer->state != RTDS_TIMER_CANCELLED ) )
            {
            RTDS_SIMULATOR_TRACE( RTDS_timerCancelled, RTDS_timer, NULL, RTDS_currentContext );
            
            /* Let's try to cancel the watchdog */
            if( wdCancel( RTDS_timer->watchDogId ) == OK )
                {
                /* Then we can delete the watchdog */
                if( wdDelete( RTDS_timer->watchDogId ) == ERROR )
                    {
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_DELETE );
                    }
                /* Remove it from the list */
                /* Is the first of the list */
                if( RTDS_prevTimer == NULL )
                    {
                    *pTimerStateList = RTDS_timer->next;
                    }
                /* Not the first of the list */
                else
                    {
                    RTDS_prevTimer->next = RTDS_timer->next;
                    }
                RTDS_FREE( RTDS_timer );
                }
            /* Could not cancel the timer. Probably went off allready */
            /* Set it cancelled in the list */
            else
                {
                RTDS_timer->state = RTDS_TIMER_CANCELLED;
                }
            return;
            }
        RTDS_prevTimer = RTDS_timer;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for( RTDS_message = RTDS_currentContext->readSaveQueue; RTDS_message != NULL; RTDS_message = RTDS_message->next )
        {
        if( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->readSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            RTDS_FREE( RTDS_message );
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    
    /* If execution gets here: the timer might be in the save queue */
    RTDS_prevMessage = NULL;
    for( RTDS_message = RTDS_currentContext->writeSaveQueue; RTDS_message != NULL; RTDS_message = RTDS_message->next )
        {
        if( RTDS_message->messageNumber == timerNumber )
            {
            /* Remove it from the list */
            /* Is the first of the list */
            if( RTDS_prevMessage == NULL )
                {
                RTDS_currentContext->writeSaveQueue = RTDS_message->next;
                }
            /* Not the first of the list */
            else
                {
                RTDS_prevMessage->next = RTDS_message->next;
                }
            RTDS_FREE( RTDS_message );
            return;
            }
        RTDS_prevMessage = RTDS_message;
        }
    }

/* **************************************************************** *
 * RTDS_GetProcessInstanceId
 * **************************************************************** *
 * Returns the instance Id of process based on its number
 * **************************************************************** *
 * Parameters:
 *      process name as a number defined in RTDS_gen.h
 * Returns:
 *      the instance id of the process if found
 * Error:
 *      System error call if not found and NULL returned
 * **************************************************************** */
RTDS_SdlInstanceId * RTDS_GetProcessInstanceId( int processNumber )
    {
    RTDS_GlobalProcessInfo * processInfo;
    RTDS_SdlInstanceId * foundInstance = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        if( processInfo->sdlProcessNumber == processNumber )
            {
            foundInstance = processInfo->mySdlInstanceId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if( foundInstance == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_PROCESS_QUEUE_ID );
        }
    
    return foundInstance;
    }

/* **************************************************************** *
 * RTDS_MsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue
 * **************************************************************** *
 * Parameters:
 *      messageNumber representing a message name
 *      dataLength length of data pointed by pData
 *      pData pointer on data sent with the message
 *      receiver instance id
 *      sender instance id
 *      RTDS_currentContext currentContext of the task
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_MsgQueueSend
( long messageNumber,
 long dataLength,
 unsigned char * pData,
 RTDS_SdlInstanceId * receiver,
 RTDS_SdlInstanceId * sender,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_MessageHeader RTDS_messageToSend;
    
    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    RTDS_messageToSend.receiver = receiver;
    RTDS_messageToSend.dataLength = dataLength;
    RTDS_messageToSend.pData = pData;
    RTDS_messageToSend.next = NULL;
    
    #ifdef RTDS_SIMULATOR
        RTDS_messageToSend.messageUniqueId = RTDS_GetMessageUniqueId();
        RTDS_messageDataToString( &RTDS_globalPrintableParameters, messageNumber, dataLength, ( void * )pData, RTDS_PARAM_CODEC_MAX_DEPTH );
        RTDS_SIMULATOR_TRACE( RTDS_messageSent, &RTDS_messageToSend, receiver, RTDS_currentContext );
        RTDS_FREE( RTDS_globalPrintableParameters );
        RTDS_globalPrintableParameters = NULL;
    #endif
    
    if( msgQSend( receiver->queueId, ( char * )&RTDS_messageToSend, sizeof( RTDS_MessageHeader ), NO_WAIT, MSG_PRI_NORMAL ) == ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }

#ifdef RTDS_SIMULATOR

/* **************************************************************** *
 * RTDS_SimulatorMsgQueueSend
 * **************************************************************** *
 * Send a message in a process's queue from the simulator
 * Basically the same as RTDS_MsgQueueSend but it gets the
 * MsgUniqueId as a parameter and does not call the dummy trace
 * function
 * **************************************************************** *
 * Parameters:
 *          messageNumber representing a message name
 *          dataLength length of data pointed by pData
 *          pData pointer on data sent with the message
 *          receiver instance id
 *          sender instance id
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_SimulatorMsgQueueSend
( long messageNumber,
 long dataLength,
 unsigned char * pData,
 RTDS_SdlInstanceId * receiver,
 RTDS_SdlInstanceId * sender,
 long MsgUniqueId )
    {
    RTDS_MessageHeader RTDS_messageToSend;
    
    RTDS_messageToSend.messageNumber = messageNumber;
    RTDS_messageToSend.timerUniqueId = 0;
    RTDS_messageToSend.sender = sender;
    
    if( dataLength != 0 )
        {
        RTDS_stringToMessageData( messageNumber, &( RTDS_messageToSend.dataLength ), ( void ** )&( RTDS_messageToSend.pData ), ( char * )pData );
        RTDS_FREE( pData );
        }
    else
        {
        RTDS_messageToSend.dataLength = 0;
        RTDS_messageToSend.pData = NULL;
        }
    
    RTDS_messageToSend.next = NULL;
    RTDS_messageToSend.messageUniqueId = MsgUniqueId;
    
    /* Specific for the backTrace since the original trace comes from the SDL-RT debugger */
    #ifdef RTDS_BACK_TRACE_MAX_EVENT_NUM
        RTDS_globalTraceEntry.event = RTDS_messageSent;
        RTDS_globalTraceEntry.eventParameter1 = ( void * )&RTDS_messageToSend;
        RTDS_globalTraceEntry.eventParameter2 = ( long )receiver;
        RTDS_globalTraceEntry.currentContext = ( RTDS_GlobalProcessInfo * )NULL;
        RTDS_TraceAdd();
    #endif
    
    if( msgQSend( receiver->queueId, ( char * )&RTDS_messageToSend, sizeof( RTDS_MessageHeader ), NO_WAIT, MSG_PRI_NORMAL ) == ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_SEND );
        }
    }
#endif

/* **************************************************************** *
 * RTDS_MsgReceive
 * **************************************************************** *
 * Receives a message from a process's queue
 * **************************************************************** *
 * Parameters:
 *          instance id wich contains queue id to read the messsage from
 *          pointer on the message to receive
 * Returns:
 *          Nothing but raise an error if an error occurs
  * **************************************************************** */
void RTDS_MsgReceive( RTDS_SdlInstanceId * instanceId, char * message )
    {
    if( RTDS_MsgQueueReceive( instanceId->queueId, message, sizeof( RTDS_MessageHeader ), WAIT_FOREVER ) == RTDS_ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_RECEIVE );
        }
    }

/* **************************************************************** *
 * RTDS_MsgQueueReceive
 * **************************************************************** *
 * Receives a message from a process's queue
 * **************************************************************** *
 * Parameters:
 *          queue id to read the messsage from
 *          pointer on the message to receive
 *          size of message
 *          mode used to read (blocking or non blocking)
 * Returns:
 *          RTDS_OK if everything goes well, RTDS_ERROR otherwise
  * **************************************************************** */
int RTDS_MsgQueueReceive( RTDS_RtosQueueId queueId, char * message, int size, int mode )
    {
    if( msgQReceive( queueId, message, size, mode ) == ERROR )
        {
        return RTDS_ERROR;
        }
    return RTDS_OK;
    }

/* **************************************************************** *
 * RTDS_ProcessCreate
 * **************************************************************** *
 * Create a new SDL process and add a processInfo struct in the
 * process info chained list
 * **************************************************************** *
 * Parameters:
 *      name of the process
 *      process number
 *      address of the process function
 *      priority fo the process
 *      the address of the instance id to be created
 *      the instance id of the parent process
 *      RTDS_currentContext currentContext of the task
 * Returns:
 *      Nothing but updates the OFFSPRING of the caller
 * **************************************************************** */
void RTDS_ProcessCreate
( char * processName,
 int processNumber,
 void * functionAddress,
 int priority,
 RTDS_SdlInstanceId ** pOffspringId,
 RTDS_SdlInstanceId * selfId,
 short synchronization,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalProcessInfo * processInfo;
    RTDS_GlobalProcessInfo * newProcessInfo;
    
    /* Allocate and fill in a new processInfo structure */
    newProcessInfo = ( RTDS_GlobalProcessInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalProcessInfo ) );
    newProcessInfo->mySdlInstanceId = ( RTDS_SdlInstanceId * )RTDS_MALLOC( sizeof( RTDS_SdlInstanceId ) );
    if( newProcessInfo == NULL || newProcessInfo->mySdlInstanceId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_MALLOC );
        }
    
    /* Sets the process name as a number */
    newProcessInfo->sdlProcessNumber = processNumber;
    /* SDL initial state */
    newProcessInfo->sdlState = 0;

    /* Last structure of the chained list */
    newProcessInfo->next = NULL;
    newProcessInfo->parentSdlInstanceId = selfId;
    newProcessInfo->offspringSdlInstanceId = NULL;
    newProcessInfo->currentMessage = NULL;
    newProcessInfo->timerList = NULL;
    #ifdef RTDS_SIMULATOR
        newProcessInfo->priority = priority;
    #endif
    
    newProcessInfo->readSaveQueue = NULL;
    newProcessInfo->writeSaveQueue = NULL;
    
    /* New task's queue creation */
    newProcessInfo->mySdlInstanceId->queueId = RTDS_MsgQueueCreate();
    newProcessInfo->mySdlInstanceId->instanceNumber = 0;
    *pOffspringId = newProcessInfo->mySdlInstanceId;
    
    /* Take the synchronization semaphore so that the created task does not run straight away */
    if( synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_HOLD;
        }
    /* Task creation */
    newProcessInfo->myRtosTaskId = taskSpawn( processName, priority, RTDS_TASK_OPTIONS, RTDS_TASK_STACK_SIZE, ( FUNCPTR )functionAddress, ( int )newProcessInfo, 0, 0, 0, 0, 0, 0, 0, 0, 0 );
    if( newProcessInfo->myRtosTaskId == ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_TASK_SPAWN );
        }
    
    /* Add the process information to the chained list pointed by the RTDS_globalProcessInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if( RTDS_globalProcessInfo == NULL )
        {
        RTDS_globalProcessInfo = newProcessInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for( processInfo = RTDS_globalProcessInfo; processInfo->next != NULL; processInfo = processInfo->next ) ;
        processInfo->next = newProcessInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_processCreated, newProcessInfo, NULL, RTDS_currentContext );
    
    /* The newly created task can now run: RTDS_globalProcessInfo and trace are up to date. */
    if( synchronization == RTDS_HOLD )
        {
        RTDS_START_SYNCHRO_UNHOLD;
        }
    }

/* **************************************************************** *
 * RTDS_MsgQueueCreate
 * **************************************************************** *
 * Create a message queue
 * **************************************************************** *
 * Parameters:
 *        Nothing
 * Returns:
 *       The created message queue ID
 * **************************************************************** */
RTDS_RtosQueueId RTDS_MsgQueueCreate( void )
    {
    RTDS_RtosQueueId queueId;
    
    queueId = msgQCreate( RTDS_QUEUE_MAX_MSG, RTDS_QUEUE_MAX_MSG_LENGTH, MSG_Q_FIFO );
    if( queueId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_MSG_Q_CREATE );
        }
    
    return queueId;
    }

/* *******************************************************************
 * RTDS_TaskCreate
 *-------------------------------------------------------------------
 * Creates a new thread.
 * If any error is encountered RTDS_SYSTEM_ERROR is called and
 * should stop execution.
 *
 * Parameters:
 *      Name of the created thread
 *      Priority of the created thread
 *      Adresse of function to execute
 *      Parameters for the created thread
 * Return:
 *      ID of the created task 
 ******************************************************************** */
RTDS_RtosTaskId RTDS_TaskCreate( char * processName, int priority, void ( * functionAddress )( void * ), void * parameter )
    {
    RTDS_RtosTaskId processId;
    
    processId = taskSpawn( processName, priority, RTDS_TASK_OPTIONS, RTDS_TASK_STACK_SIZE, ( FUNCPTR )functionAddress, ( int )parameter, 0, 0, 0, 0, 0, 0, 0, 0, 0 );
    if( processId == ERROR )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_PROCESS_CREATE_TASK_SPAWN );
        }
    
    return processId;
    }

/* **************************************************************** *
 * RTDS_ProcessForget
 *-------------------------------------------------------------------
 * Forgets a given process, freeing all associated information
 *
 * Parameters:
 *      RTDS_currentContext : The current context of the task to delete
 *      deleteQueue: If true, the message queue for the process is destroyed
 * Returns:
 *      The PID for the forgotten process or 0 if process was not found
 * **************************************************************** */
RTDS_RtosTaskId RTDS_ProcessForget( RTDS_GlobalProcessInfo * RTDS_currentContext, short deleteQueue )
    {
    RTDS_GlobalProcessInfo * processInfo;
    RTDS_GlobalProcessInfo * previousProcessInfo;
    RTDS_TimerState * RTDS_timer;
    RTDS_MessageHeader * message;
    RTDS_MessageHeader * tmpMessage;
    RTDS_RtosTaskId pidToKill = 0;

    /* Let's free the current message if any */
    if( RTDS_currentContext->currentMessage != NULL )
        {
        #ifdef RTDS_SIMULATOR
          /* Release the message unique id back to the pool */
          RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif

        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        }
    
    /* Let's clean the timer chained list to free memory and stop watchdogs */
    for( RTDS_timer = RTDS_currentContext->timerList; RTDS_timer != NULL; RTDS_timer = RTDS_currentContext->timerList )
        {
        if( wdDelete( RTDS_timer->watchDogId ) == ERROR )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_WATCHDOG_DELETE );
            }
        RTDS_currentContext->timerList = RTDS_timer->next;
        RTDS_FREE( RTDS_timer );
        }
    
    /* Clean the save queue: free messages and message unique ids in the read and write save queues */
    for( message = RTDS_currentContext->readSaveQueue; message != NULL; )
        {
    #ifdef RTDS_SIMULATOR
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
    #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        }
    
    for( message = RTDS_currentContext->writeSaveQueue; message != NULL; )
        {
    #ifdef RTDS_SIMULATOR
        RTDS_ReleaseMessageUniqueId( message->messageUniqueId );
    #endif
        tmpMessage = message;
        message = message->next;
        RTDS_FREE( tmpMessage );
        }
    previousProcessInfo = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for( processInfo = RTDS_globalProcessInfo; processInfo != NULL; processInfo = processInfo->next )
        {
        /* The queue id is used to find the process id */
        if( processInfo->mySdlInstanceId == RTDS_currentContext->mySdlInstanceId )  /* Found ! */
            {
            /* Update the process information chained list */
            if( previousProcessInfo == NULL )    /* First one in the list */
                {
                RTDS_globalProcessInfo = processInfo->next;
                }
            else
                {
                previousProcessInfo->next = processInfo->next;
                }
            
            RTDS_SIMULATOR_TRACE( RTDS_processDied, processInfo, NULL, RTDS_currentContext );
            pidToKill = processInfo->myRtosTaskId;
            /* Delete the message queue and free the process info block */
            if ( deleteQueue )
                {
                if( msgQDelete( processInfo->mySdlInstanceId->queueId ) != OK )
                    {
                    RTDS_SYSTEM_ERROR( RTDS_ERROR_MSG_QUEUE_DELETE );
                    }
                }
            RTDS_FREE( processInfo->mySdlInstanceId );  
            RTDS_FREE( processInfo );
            break;
            }
        previousProcessInfo = processInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    return pidToKill;
    }

/* **************************************************************** *
 * RTDS_ProcessKill
 * **************************************************************** *
 * Kills an SDL process and delete its queue and process info block
 * **************************************************************** *
 * Parameters:
 *      the address of the process current context
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_ProcessKill( RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_RtosTaskId pidToKill = 0;
    
    /* Forget everything about process, delete its message queue and get its PID */
    pidToKill = RTDS_ProcessForget( RTDS_currentContext, 1 );
    
    /* Delete the task */
    if( pidToKill != 0 )
        {
        if( taskDelete( pidToKill ) != OK )
            {
            RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_DELETE );
            }
        }
    else
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TASK_TO_DELETE_NOT_FOUND );
        }
    }

/* **************************************************************** *
 * RTDS_Sem_Info_Insert
 * **************************************************************** *
 * Inserts a semaphoreInfo struct in the semaphore info chained list
 * **************************************************************** *
 * Parameters:
 *      semaphore number
 *      id of the semaphore
 *      RTDS_currentContext currentContext of the task
 * Returns:
 *      The semaphore id so that the user can use the semaphore id directly
 * **************************************************************** */
RTDS_SemaphoreId RTDS_Sem_Info_Insert
( int semaphoreNumber,
 RTDS_SemaphoreId semaphoreId,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_GlobalSemaphoreInfo * newSemInfo;
    
    /* Allocate and fill in a new semInfo structure */
    newSemInfo = ( RTDS_GlobalSemaphoreInfo * )RTDS_MALLOC( sizeof( RTDS_GlobalSemaphoreInfo ) );
    if( newSemInfo == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT_MALLOC );
        }
    
    newSemInfo->semaphoreNumber = semaphoreNumber;
    newSemInfo->next = NULL;
    
    /* Semaphore creation */
    newSemInfo->semaphoreId = semaphoreId;
    if( newSemInfo->semaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_SEM_INFO_INSERT );
        }
    
    /* Add the semaphore information to the chained list pointed by the RTDS_globalSemaphoreInfo global variable */
    RTDS_CRITICAL_SECTION_START;
    if( RTDS_globalSemaphoreInfo == NULL )
        {
        RTDS_globalSemaphoreInfo = newSemInfo;
        }
    else
        {
        /* Let's get to the end of the list */
        for( semInfo = RTDS_globalSemaphoreInfo; semInfo->next != NULL; semInfo = semInfo->next );
        semInfo->next = newSemInfo;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreCreated, semaphoreId, -1, RTDS_currentContext );
    
    return semaphoreId;
    }

/* **************************************************************** *
 * RTDS_Sem_Delete
 * **************************************************************** *
 * Kills a VxWorks semaphore and delete its info from the semaphore
 * information chained list and free the related memory
 * **************************************************************** *
 * Parameters:
 *      the address of the semaphore to find the semaphore info block
 *      RTDS_currentContext currentContext of the task
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_Sem_Delete( RTDS_SemaphoreId semaphoreId, RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_GlobalSemaphoreInfo * semInfoPrev;
    
    RTDS_SIMULATOR_TRACE( RTDS_semaphoreDeleted, semaphoreId, 0, RTDS_currentContext );
    
    /* Remove the semaphore information from the chained list */
    semInfoPrev = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
    {
    if( semInfo->semaphoreId == semaphoreId ) /* semaphore found */
        {
        if( semInfoPrev == NULL )
            {
            RTDS_globalSemaphoreInfo = semInfo->next;
            }
        else
            {
            semInfoPrev->next = semInfo->next;
            }
        RTDS_SemaphoreIdDelete( semInfo->semaphoreId );
        RTDS_FREE( semInfo );
        break;
        }
    semInfoPrev = semInfo;
    }
    RTDS_CRITICAL_SECTION_STOP;
    
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdDelete
 * **************************************************************** *
 * Kills a VxWorks semaphore
 * **************************************************************** *
 * Parameters:
 *      the id of the semaphore
 * Returns:
 *      Nothing but raise RTDS_SYSTEM_ERROR if an error occurs
 * **************************************************************** */
void RTDS_SemaphoreIdDelete( RTDS_SemaphoreId semaphoreId )
    {
    if( semDelete( semaphoreId ) != OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_DELETE_SEMAPHORE_ID );
        }
    }

/* **************************************************************** *
 * RTDS_MutexSemaphoreCreate
 * **************************************************************** *
 * Create a VxWorks mutex semaphore
 * **************************************************************** *
 * Parameters:
 *      Mutex semaphore options
 * Returns:
 *      The created semaphore id
 * **************************************************************** */
RTDS_SemaphoreId RTDS_MutexSemaphoreCreate( int options )
    {
    RTDS_SemaphoreId semMid;
    
    if( ( semMid = semMCreate( options ) ) == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_MUTEX_SEMAPHORE );
        }
    
    return semMid;
    }

/* **************************************************************** *
 * RTDS_BinarySemaphoreCreate
 * **************************************************************** *
 * Create a VxWorks binary semaphore
 * **************************************************************** *
 * Parameters:
 *      semaphore options
 *      initial semaphore state
 * Returns:
 *      The created semaphore id
 * **************************************************************** */
RTDS_SemaphoreId RTDS_BinarySemaphoreCreate( int options, SEM_B_STATE initialState )
    {
    RTDS_SemaphoreId semMid;
    
    if( ( semMid = semBCreate( options, initialState ) ) == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_CREATE_BINARY_SEMAPHORE );
        }
    
    return semMid;
    }

/* **************************************************************** *
 * RTDS_TakeSemaphoreId
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *      the id of the semaphore
 *      timeout value
 *      RTDS_currentContext currentContext of the task
 * Returns:
 *      status
 * **************************************************************** */
RTDS_SemaphoreStatus RTDS_TakeSemaphoreId
( RTDS_SemaphoreId semaphoreId,
 RTDS_SemaphoreTimeout TIME_OUT,
 RTDS_GlobalProcessInfo * RTDS_currentContext )
    {
    RTDS_SemaphoreStatus tmpStatus;
    
    RTDS_SIMULATOR_TRACE( RTDS_semTakeAttempt, semaphoreId, TIME_OUT, RTDS_currentContext );
    
    tmpStatus = semTake( semaphoreId, TIME_OUT );
    if( tmpStatus == OK )
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeSucceded, semaphoreId, -1, RTDS_currentContext );
        }
    else
        {
        RTDS_SIMULATOR_TRACE( RTDS_semTakeTimedOut, semaphoreId, -1, RTDS_currentContext );
        }
    
    return tmpStatus;
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdGive
 * **************************************************************** *
 * Attempt to give a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *      the id of the semaphore
 * Returns:
 *      nothing or raise an error
 * **************************************************************** */
void RTDS_SemaphoreIdGive( RTDS_SemaphoreId semaphoreId )
    {
    if( semGive( semaphoreId ) != OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GIVE_SEMAPHORE_ID );
        }
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdFlush
 * **************************************************************** *
 * Attempt to unblock every task pended on a specific semaphore id
 * **************************************************************** *
 * Parameters:
 *      the id of the semaphore
 * Returns:
 *      nothing or raise an error
 * **************************************************************** */
void RTDS_SemaphoreIdFlush( RTDS_SemaphoreId semaphoreId )
    {
    if( semFlush( semaphoreId ) != OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_FLUSH_SEMAPHORE_ID );
        }
    }

/* **************************************************************** *
 * RTDS_SemaphoreIdTake
 * **************************************************************** *
 * Attempt to take a semaphore from its id
 * **************************************************************** *
 * Parameters:
 *      the id of the semaphore
  *     timeout value
 * Returns:
 *      nothing or raise an error
 * **************************************************************** */
void RTDS_SemaphoreIdTake( RTDS_SemaphoreId semaphoreId, RTDS_SemaphoreTimeout TIME_OUT )
    {
    if( semTake( semaphoreId, TIME_OUT ) != OK )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_TAKE_SEMAPHORE_ID );
        }
    }

/* **************************************************************** *
 * RTDS_GetSemaphoreId
 * **************************************************************** *
 * Gets the id of a semaphore from its number (name)
 * **************************************************************** *
 * Parameters:
 *      semaphore number representing its name
 * Returns:
 *      the id of the semaphore
 * **************************************************************** */
RTDS_SemaphoreId RTDS_GetSemaphoreId( int semaphoreNumber )
    {
    RTDS_GlobalSemaphoreInfo * semInfo;
    RTDS_SemaphoreId foundSemaphoreId;
    
    foundSemaphoreId = NULL;
    
    RTDS_CRITICAL_SECTION_START;
    for( semInfo = RTDS_globalSemaphoreInfo; semInfo != NULL; semInfo = semInfo->next )
        {
        if( semInfo->semaphoreNumber == semaphoreNumber ) /* semaphore found */
            {
            foundSemaphoreId = semInfo->semaphoreId;
            break;
            }
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    if( foundSemaphoreId == NULL )
        {
        RTDS_SYSTEM_ERROR( RTDS_ERROR_GET_SEMAPHORE_ID );
        }
        
    return foundSemaphoreId;    /* Might cause an RTOS exception if NULL*/
    }

/* **************************************************************** *
 * RTDS_TimerCleanUp
 *-------------------------------------------------------------------
 * Called when the last received message is a timer to figure out if
 * timer was not cancelled before it was received.
 *
 * Parameters:
 *     currentContext : RTDS_GlobalProcessInfo* context of the current task
 * **************************************************************** */
void RTDS_TimerCleanUp( RTDS_GlobalProcessInfo * currentContext )
    {
    RTDS_TimerState * timer;
    RTDS_TimerState * prevTimer;
    
    prevTimer = NULL;
    for( timer = currentContext->timerList; timer != NULL; timer = timer->next )
        {
        /* If timer found and cancelled */
        if( timer->timerUniqueId == currentContext->currentMessage->timerUniqueId )
            {
            if( timer->state == RTDS_TIMER_CANCELLED )
                {
                /* Discard current message */
                RTDS_FREE( currentContext->currentMessage );
                currentContext->currentMessage = NULL;
                }
            
            /* Remove it from list of timers */
            if( prevTimer == NULL )
                {
                currentContext->timerList = currentContext->timerList->next;
                }
            else
                {
                prevTimer->next = timer->next;
                }
            RTDS_FREE( timer );
            break;
            }
        prevTimer = timer;
        }
    }

/* **************************************************************** *
 * RTDS_TransitionCleanUp
 * **************************************************************** *
 * Called at the end of transitions:
 * frees message buffer if valid
 * re-organize save queue if state has changed
 * **************************************************************** *
  * Parameters:
 *      current context of the task
 *      Previous SDL state
 * Returns:
 *      Nothing
 * **************************************************************** */
void RTDS_TransitionCleanUp( RTDS_GlobalProcessInfo * RTDS_currentContext, int RTDS_sdlStatePrev )
    {
    /* Free message buffer if valid */
    if( RTDS_currentContext->currentMessage != NULL )
        {
        /* Free the message parameter structure if any */
        if (RTDS_currentContext->currentMessage->pData != NULL){
          RTDS_FREE(RTDS_currentContext->currentMessage->pData);
          }

        #ifdef RTDS_SIMULATOR
            /* Release the message unique id back to the pool */
            RTDS_ReleaseMessageUniqueId( RTDS_currentContext->currentMessage->messageUniqueId );
        #endif
        
        /* Free memory */
        RTDS_FREE( RTDS_currentContext->currentMessage );
        RTDS_currentContext->currentMessage = NULL;
        } /* End of if( RTDS_currentContext->currentMessage != NULL ) */
    
    /* If SDL state has changed and messages have been saved: reorganise the save queue */
    if( ( RTDS_currentContext->sdlState != RTDS_sdlStatePrev ) && ( RTDS_currentContext->writeSaveQueue != NULL ) )
        {
        RTDS_MessageHeader * message;
        /* Let's get to the end of the save queue */
        for( message=RTDS_currentContext->writeSaveQueue; message->next != NULL; message = message->next );
        message->next = RTDS_currentContext->readSaveQueue;
        RTDS_currentContext->readSaveQueue = RTDS_currentContext->writeSaveQueue;
        RTDS_currentContext->writeSaveQueue = NULL;
        }
    return;
    }

/* **************************************************************** *
 * RTDS_GetSystemTime
 * **************************************************************** *
 * As its name states...
 * **************************************************************** *
 * Parameters:
 *     None
 * Returns:
 *     System tick count value
 * **************************************************************** */
unsigned long RTDS_GetSystemTime( void )
    {
    return tickGet();
    }

/* **************************************************************** *
 * **************************************************************** *
 * **************************************************************** *
 * THE CODE BELOW IS ONLY USED TO DEBUG WITH RTDS SDL-RT DEBUGGER
 * **************************************************************** *
 * **************************************************************** *
 * **************************************************************** */

#ifdef RTDS_SIMULATOR

unsigned char * RTDS_globalMessageUniqueIdPool = NULL;
RTDS_GlobalTraceInfo RTDS_globalTraceEntry = { RTDS_systemError, NULL, 0 };
char * RTDS_globalPrintableParameters = NULL;

/* **************************************************************** *
 * RTDS_GetMessageUniqueId
 * **************************************************************** *
 * Gets a message unique id for the simulator
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      the message unique id (minimum is 1)
 * **************************************************************** */
unsigned long RTDS_GetMessageUniqueId( void )
    {
    unsigned char * index;
    long uniqueByteId;
    long uniqueBitId;
    
    index = RTDS_globalMessageUniqueIdPool;
    
    RTDS_CRITICAL_SECTION_START;
    for( uniqueByteId = 0; uniqueByteId < RTDS_MESSAGE_UNIQUE_ID_POOL_SIZE; uniqueByteId++ )
        {
        if( *index != 0xFF )
            {
            for( uniqueBitId = 0; uniqueBitId < 8; uniqueBitId++ )
                {
                if( ( ( 1 << uniqueBitId ) & *index ) == 0 )
                    {
                    *index = *index | ( 1 << uniqueBitId );
                    RTDS_CRITICAL_SECTION_STOP;
                    return ( 8 * uniqueByteId + uniqueBitId + 1 );
                    }
                }
            }
        index++;
        }
    RTDS_CRITICAL_SECTION_STOP;
    
    /* All bits are set... No more message unique id */
    RTDS_SYSTEM_ERROR( RTDS_ERROR_NO_MORE_MSG_UNIQUE_ID );
    
    return 0;
    }

/* **************************************************************** *
 * RTDS_ReleaseMessageUniqueId
 * **************************************************************** *
 * Make a message unique id available from the pool
 * **************************************************************** *
 * Parameters:
 *      message unique id
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_ReleaseMessageUniqueId( unsigned long messageUniqueId )
    {
    unsigned char * index;
    
    if( messageUniqueId == 0 ) /* probably a timer */
        {
        return;
        }
    messageUniqueId -= 1;
    index = RTDS_globalMessageUniqueIdPool;
    index += ( unsigned char )( messageUniqueId / 8 );
    
    RTDS_CRITICAL_SECTION_START;
    ( *index ) = ( *index ) & ~( 1 << messageUniqueId % 8 );
    RTDS_CRITICAL_SECTION_STOP;
    
    }

/* **************************************************************** *
 * RTDS_MemAlloc
 * **************************************************************** *
 * Allocates memory
 * To be called by the SDL-RT debugger: useless in final application
 * **************************************************************** *
 * Parameters:
 *      size of the memory block to allocate
 * Returns:
 *      a pointer on the newly allocated block
 * **************************************************************** */
void * RTDS_MemAlloc( size_t size )
    {
    return malloc( size );
    }

/* **************************************************************** *
 * RTDS_DummyTraceFunction
 * **************************************************************** *
 * As its name states... The simulator sets a breakpoint on this
 * function and reads the RTDS_globalTraceEntry variable to see
 * what happened
 * **************************************************************** *
 * Parameters:
 *      None
 * Returns:
 *      nothing
 * **************************************************************** */
void RTDS_DummyTraceFunction( void )
    {
    }

#endif
